/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * This file incorporates work covered by the following license notice:
 *
 *   Licensed to the Apache Software Foundation (ASF) under one or more
 *   contributor license agreements. See the NOTICE file distributed
 *   with this work for additional information regarding copyright
 *   ownership. The ASF licenses this file to you under the Apache
 *   License, Version 2.0 (the "License"); you may not use this file
 *   except in compliance with the License. You may obtain a copy of
 *   the License at http://www.apache.org/licenses/LICENSE-2.0 .
 */

#pragma once

#include "utility.hxx"

#include <string_view>
#include <rtl/ustring.hxx>
#include <svl/SfxBroadcaster.hxx>
#include <unotools/configitem.hxx>

#include "types.hxx"

namespace com::sun::star::uno
{
template <class E> class Sequence;
}

class SmSym;
class SmSymbolManager;
class SmFormat;
namespace vcl
{
class Font;
}
struct SmCfgOther;
class SfxItemSet;

struct SmFontFormat
{
    OUString aName;
    sal_Int16 nCharSet;
    sal_Int16 nFamily;
    sal_Int16 nPitch;
    sal_Int16 nWeight;
    sal_Int16 nItalic;

    SmFontFormat();
    explicit SmFontFormat(const vcl::Font& rFont);

    vcl::Font GetFont() const;
    bool operator==(const SmFontFormat& rFntFmt) const;
};

struct SmFntFmtListEntry
{
    OUString aId;
    SmFontFormat aFntFmt;

    SmFntFmtListEntry(OUString aId, SmFontFormat rFntFmt);
};

class SmFontFormatList
{
    std::vector<SmFntFmtListEntry> aEntries;
    bool bModified;

    SmFontFormatList(const SmFontFormatList&) = delete;
    SmFontFormatList& operator=(const SmFontFormatList&) = delete;

public:
    SmFontFormatList();

    void Clear();
    void AddFontFormat(const OUString& rFntFmtId, const SmFontFormat& rFntFmt);
    void RemoveFontFormat(std::u16string_view rFntFmtId);

    const SmFontFormat* GetFontFormat(std::u16string_view rFntFmtId) const;
    const SmFontFormat* GetFontFormat(size_t nPos) const;
    OUString GetFontFormatId(const SmFontFormat& rFntFmt) const;
    OUString GetFontFormatId(const SmFontFormat& rFntFmt, bool bAdd);
    OUString GetFontFormatId(size_t nPos) const;
    OUString GetNewFontFormatId() const;
    size_t GetCount() const { return aEntries.size(); }

    bool IsModified() const { return bModified; }
    void SetModified(bool bVal) { bModified = bVal; }
};

class SmMathConfig final : public utl::ConfigItem, public SfxBroadcaster
{
    std::unique_ptr<SmFormat> pFormat;
    std::unique_ptr<SmCfgOther> pOther;
    std::unique_ptr<SmFontFormatList> pFontFormatList;
    std::unique_ptr<SmSymbolManager> pSymbolMgr;
    css::uno::Sequence<OUString> m_sUserDefinedNames;
    bool bIsOtherModified;
    bool bIsFormatModified;
    SmFontPickList vFontPickList[8];
    sal_Int32 m_nCommitLock = 0;

    SmMathConfig(const SmMathConfig&) = delete;
    SmMathConfig& operator=(const SmMathConfig&) = delete;

    void StripFontFormatList(const std::vector<SmSym>& rSymbols);

    void Save();

    void ReadSymbol(SmSym& rSymbol, const OUString& rSymbolName,
                    std::u16string_view rBaseNode) const;
    void ReadFontFormat(SmFontFormat& rFontFormat, std::u16string_view rSymbolName,
                        std::u16string_view rBaseNode) const;

    bool SetOtherIfNotEqual(bool& rbItem, bool bNewVal);

    void LoadOther();
    void SaveOther();
    void LoadFormat();
    void SaveFormat();
    void LoadFontFormatList();
    void SaveFontFormatList();

    void SetOtherModified(bool bVal);
    bool IsOtherModified() const { return bIsOtherModified; }
    void SetFormatModified(bool bVal);
    bool IsFormatModified() const { return bIsFormatModified; }

    SmFontFormatList& GetFontFormatList();
    const SmFontFormatList& GetFontFormatList() const
    {
        return const_cast<SmMathConfig*>(this)->GetFontFormatList();
    }

    virtual void ImplCommit() override;
    void LockCommit() { ++m_nCommitLock; }
    void UnlockCommit();
    // Used to avoid tens of atomic commits in e.g. ItemSetToConfig that calls individual setters
    friend struct CommitLocker;
    struct CommitLocker
    {
        SmMathConfig& m_rConfig;
        CommitLocker(SmMathConfig& rConfig)
            : m_rConfig(rConfig)
        {
            m_rConfig.LockCommit();
        }
        ~CommitLocker() { m_rConfig.UnlockCommit(); }
    };

    void Clear();

public:
    SmMathConfig();
    virtual ~SmMathConfig() override;

    // utl::ConfigItem
    virtual void Notify(const css::uno::Sequence<OUString>& rPropertyNames) override;

    SmSymbolManager& GetSymbolManager();
    void GetSymbols(std::vector<SmSym>& rSymbols) const;
    void SetSymbols(const std::vector<SmSym>& rNewSymbols);

    const SmFormat& GetStandardFormat() const;
    void SetStandardFormat(const SmFormat& rFormat, bool bSaveFontFormatList = false);

    css::uno::Sequence<OUString> LoadUserDefinedNames();
    void GetUserDefinedFormula(std::u16string_view sName, OUString& sFormula);
    bool HasUserDefinedFormula(std::u16string_view sName);
    void SaveUserDefinedFormula(std::u16string_view sName, const OUString& sElement);
    void DeleteUserDefinedFormula(std::u16string_view sName);

    bool IsPrintTitle() const;
    void SetPrintTitle(bool bVal);
    bool IsPrintFormulaText() const;
    void SetPrintFormulaText(bool bVal);
    bool IsPrintFrame() const;
    void SetPrintFrame(bool bVal);
    SmPrintSize GetPrintSize() const;
    void SetPrintSize(SmPrintSize eSize);
    sal_uInt16 GetPrintZoomFactor() const;
    void SetPrintZoomFactor(sal_uInt16 nVal);
    sal_uInt16 GetSmEditWindowZoomFactor() const;
    void SetSmEditWindowZoomFactor(sal_uInt16 nVal);

    bool IsSaveOnlyUsedSymbols() const;
    void SetSaveOnlyUsedSymbols(bool bVal);
    bool IsAutoCloseBrackets() const;
    void SetAutoCloseBrackets(bool bVal);
    bool IsInlineEditEnable() const;
    void SetInlineEditEnable(bool bVal);
    bool IsIgnoreSpacesRight() const;
    void SetIgnoreSpacesRight(bool bVal);
    bool IsAutoRedraw() const;
    void SetAutoRedraw(bool bVal);
    bool IsShowFormulaCursor() const;
    void SetShowFormulaCursor(bool bVal);

    sal_Int16 GetDefaultSmSyntaxVersion() const;
    void SetDefaultSmSyntaxVersion(sal_Int16 nVal);

    SmFontPickList& GetFontPickList(sal_uInt16 nIdent) { return vFontPickList[nIdent]; }

    void ItemSetToConfig(const SfxItemSet& rSet);
    void ConfigToItemSet(SfxItemSet& rSet) const;
};

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
