﻿' Licensed to the .NET Foundation under one or more agreements.
' The .NET Foundation licenses this file to you under the MIT license.
' See the LICENSE file in the project root for more information.

Imports System.Collections.Immutable
Imports System.IO
Imports Microsoft.CodeAnalysis
Imports Microsoft.CodeAnalysis.SpecialType
Imports Microsoft.CodeAnalysis.Test.Utilities
Imports Microsoft.CodeAnalysis.Text
Imports Microsoft.CodeAnalysis.VisualBasic
Imports Microsoft.CodeAnalysis.VisualBasic.OverloadResolution
Imports Microsoft.CodeAnalysis.VisualBasic.Symbols
Imports Microsoft.CodeAnalysis.VisualBasic.Syntax

Imports Roslyn.Test.Utilities

Namespace Microsoft.CodeAnalysis.VisualBasic.UnitTests
    Partial Public Class GetExtendedSemanticInfoTests : Inherits SemanticModelTestBase

        <Fact>
        Public Sub LambdaInInitializer()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
            Class C
            Private Shared AmbiguousInNSError As Func(Of C, D) =
                Function(syms As C) As D
                    If C IsNot Nothing
                        Return New D()'BIND:"D"
                    Else
                        Return New D()
                    End If
                End Function
            End Class
    </file>
    </compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")
            'simply not crashing is the goal for now.
        End Sub

        <Fact>
        Public Sub BindLambdasInArgsOfBadParent()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
'note that T is not defined which causes the object creation expression
'to be bad. This test ensures that the arguments are still bound and analyzable.
        Module M
        Private Shared Function Meth() As T
            Return New T(Function() String.Empty)'BIND:"String"
        End Function
        End Module
    </file>
    </compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of PredefinedTypeSyntax)(compilation, "a.vb")
            'not crashing 
        End Sub

        <Fact>
        Public Sub RunningAfoulOfExtensions()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
        Imports System.Runtime.CompilerServices 
        Imports System.Linq.Expressions
        Module M
        Private Shared Function GetExpressionType(x As Symbol) As TypeSymbol
            Select Case x.Kind
                Case Else
                    Dim type = TryCast(x, TypeSymbol)'BIND:"x"
                    If type IsNot Nothing Then
                        Return type
                    End If
            End Select

            Return Nothing
        End Function
            &lt;Extension()&gt;
            Sub Type(ByVal x As Integer)
            End Sub
            &lt;Extension()&gt;
            Sub Type(x As String)
            End Sub
        End Module
    </file>
    </compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")
            'not crashing 
        End Sub

        <Fact>
        Public Sub BindPredefinedTypeOutsideMethodBody()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
Imports System
Imports System.Collections.Generic
Imports System.Linq

Module Program
    Sub Main(args As String())'BIND:"String"

    End Sub
End Module
    </file>
    </compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of PredefinedTypeSyntax)(compilation, "a.vb")

            Assert.Equal("System.String", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("System.String", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("System.String", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.NamedType, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub BindPredefinedTypeInsideMethodBody()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
Imports System
Imports System.Collections.Generic
Imports System.Linq

Module Program
    Sub Main(args As String())
        Dim x = String.Empty 'BIND:"String"

    End Sub
End Module
    </file>
    </compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of PredefinedTypeSyntax)(compilation, "a.vb")

            Assert.Equal("System.String", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("System.String", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("System.String", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.NamedType, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub ConvertedLocal()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System
Imports System.Collections.Generic
Imports System.Linq

Module Program
    Sub Main(args As String())
        Dim x As Integer
        Dim y As Long
        y = x'BIND:"x"

    End Sub
End Module
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("System.Int32", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Int64", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.WideningNumeric, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("x As System.Int32", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Local, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub InaccessibleType1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System

Module Program
    Dim a As A.B'BIND:"B"
    Sub Main(args As String())
    End Sub
End Module

Class A
    Private Class B
        Public Sub f()

        End Sub
    End Class
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("A.B", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("A.B", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Null(semanticInfo.Symbol)
            Assert.Equal(CandidateReason.Inaccessible, semanticInfo.CandidateReason)
            Assert.Equal(1, semanticInfo.CandidateSymbols.Length)
            Dim sortedCandidates = semanticInfo.CandidateSymbols.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("A.B", sortedCandidates(0).ToTestDisplayString())
            Assert.Equal(SymbolKind.NamedType, sortedCandidates(0).Kind)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub InaccessibleType2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System

Module Program
    Sub Main(args As String())
        A.B.f()'BIND:"B"
    End Sub
End Module

Class A
    Private Class B
        Public Sub f()

        End Sub
    End Class
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("A.B", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("A.B", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Null(semanticInfo.Symbol)
            Assert.Equal(CandidateReason.Inaccessible, semanticInfo.CandidateReason)
            Assert.Equal(1, semanticInfo.CandidateSymbols.Length)
            Dim sortedCandidates = semanticInfo.CandidateSymbols.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("A.B", sortedCandidates(0).ToTestDisplayString())
            Assert.Equal(SymbolKind.NamedType, sortedCandidates(0).Kind)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub InaccessibleType3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System

Module Program
    Sub Main(args As String())
        Dim z As A.B'BIND:"B"
    End Sub
End Module

Class A
    Private Class B
        Public Sub f()

        End Sub
    End Class
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("A.B", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("A.B", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Null(semanticInfo.Symbol)
            Assert.Equal(CandidateReason.Inaccessible, semanticInfo.CandidateReason)
            Assert.Equal(1, semanticInfo.CandidateSymbols.Length)
            Dim sortedCandidates = semanticInfo.CandidateSymbols.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("A.B", sortedCandidates(0).ToTestDisplayString())
            Assert.Equal(SymbolKind.NamedType, sortedCandidates(0).Kind)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub AmbiguousType1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System
Imports N1, N2

Module Program
    Dim x As A'BIND:"A"
    Sub Main(args As String())

    End Sub
End Module

Namespace N1
    Class A
    End Class
End Namespace

Namespace N2
    Class A

    End Class
End Namespace
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("A", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Error, semanticInfo.Type.TypeKind)
            Assert.Equal("A", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Error, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Null(semanticInfo.Symbol)
            Assert.Equal(CandidateReason.Ambiguous, semanticInfo.CandidateReason)
            Assert.Equal(2, semanticInfo.CandidateSymbols.Length)
            Dim sortedCandidates = semanticInfo.CandidateSymbols.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("N1.A", sortedCandidates(0).ToTestDisplayString())
            Assert.Equal(SymbolKind.NamedType, sortedCandidates(0).Kind)
            Assert.Equal("N2.A", sortedCandidates(1).ToTestDisplayString())
            Assert.Equal(SymbolKind.NamedType, sortedCandidates(1).Kind)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub AmbiguousType2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System
Imports N1, N2

Module Program
    Sub Main(args As String())
        Dim x As A'BIND:"A"
    End Sub
End Module

Namespace N1
    Class A
    End Class
End Namespace

Namespace N2
    Class A

    End Class
End Namespace
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("A", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Error, semanticInfo.Type.TypeKind)
            Assert.Equal("A", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Error, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Null(semanticInfo.Symbol)
            Assert.Equal(CandidateReason.Ambiguous, semanticInfo.CandidateReason)
            Assert.Equal(2, semanticInfo.CandidateSymbols.Length)
            Dim sortedCandidates = semanticInfo.CandidateSymbols.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("N1.A", sortedCandidates(0).ToTestDisplayString())
            Assert.Equal(SymbolKind.NamedType, sortedCandidates(0).Kind)
            Assert.Equal("N2.A", sortedCandidates(1).ToTestDisplayString())
            Assert.Equal(SymbolKind.NamedType, sortedCandidates(1).Kind)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub AmbiguousType3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System
Imports N1, N2

Module Program
    Sub Main(args As String())
        A.goo()'BIND:"A"
    End Sub
End Module

Namespace N1
    Class A
    End Class
End Namespace

Namespace N2
    Class A

    End Class
End Namespace
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("A", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Error, semanticInfo.Type.TypeKind)
            Assert.Equal("A", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Error, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Null(semanticInfo.Symbol)
            Assert.Equal(CandidateReason.Ambiguous, semanticInfo.CandidateReason)
            Assert.Equal(2, semanticInfo.CandidateSymbols.Length)
            Dim sortedCandidates = semanticInfo.CandidateSymbols.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("N1.A", sortedCandidates(0).ToTestDisplayString())
            Assert.Equal(SymbolKind.NamedType, sortedCandidates(0).Kind)
            Assert.Equal("N2.A", sortedCandidates(1).ToTestDisplayString())
            Assert.Equal(SymbolKind.NamedType, sortedCandidates(1).Kind)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub InaccessibleSharedField()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System


Module Program
    Sub Main(args As String())
        Dim z As Integer = A.fi'BIND:"fi"
    End Sub
End Module

Class A
    Private Shared fi As Integer
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("System.Int32", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Int32", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Null(semanticInfo.Symbol)
            Assert.Equal(CandidateReason.Inaccessible, semanticInfo.CandidateReason)
            Assert.Equal(1, semanticInfo.CandidateSymbols.Length)
            Dim sortedCandidates = semanticInfo.CandidateSymbols.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("A.fi As System.Int32", sortedCandidates(0).ToTestDisplayString())
            Assert.Equal(SymbolKind.Field, sortedCandidates(0).Kind)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub MethodGroup1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System

Module Program
    Sub Main(args As String())
        f(3)'BIND:"f"
    End Sub

    Sub f()
    End Sub

    Sub f(x As Integer)
    End Sub

    Function f(ByVal a As Integer, b As Long) As String
        Return "hi"
    End Function
End Module
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Null(semanticInfo.Type)
            Assert.Null(semanticInfo.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Sub Program.f(x As System.Int32)", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(3, semanticInfo.MemberGroup.Length)
            Dim sortedMethodGroup = semanticInfo.MemberGroup.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Function Program.f(a As System.Int32, b As System.Int64) As System.String", sortedMethodGroup(0).ToTestDisplayString())
            Assert.Equal("Sub Program.f()", sortedMethodGroup(1).ToTestDisplayString())
            Assert.Equal("Sub Program.f(x As System.Int32)", sortedMethodGroup(2).ToTestDisplayString())

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub MethodGroup2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System

Module Program
    Sub Main(args As String())
        Dim x As New Class1()
        x.f(3, 7)'BIND:"x.f"
    End Sub

End Module

Class Class1
    Public Sub f()
    End Sub

    Public Sub f(x As Integer)
    End Sub

    Public Function f(ByVal a As Integer, b As Long) As String
        Return "hi"
    End Function
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of MemberAccessExpressionSyntax)(compilation, "a.vb")

            Assert.Null(semanticInfo.Type)
            Assert.Null(semanticInfo.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Function Class1.f(a As System.Int32, b As System.Int64) As System.String", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(3, semanticInfo.MemberGroup.Length)
            Dim sortedMethodGroup = semanticInfo.MemberGroup.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Function Class1.f(a As System.Int32, b As System.Int64) As System.String", sortedMethodGroup(0).ToTestDisplayString())
            Assert.Equal("Sub Class1.f()", sortedMethodGroup(1).ToTestDisplayString())
            Assert.Equal("Sub Class1.f(x As System.Int32)", sortedMethodGroup(2).ToTestDisplayString())

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub InaccessibleMethodGroup()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System

Module Program
    Sub Main(args As String())
        Dim x As New Class1()
        x.f(3, 7)'BIND:"x.f"
    End Sub

End Module

Class Class1
    Protected Sub f()
    End Sub

    Protected Sub f(x As Integer)
    End Sub

    Private Function f(ByVal a As Integer, b As Long) As String
        Return "hi"
    End Function
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of MemberAccessExpressionSyntax)(compilation, "a.vb")

            ' This test should return all three inaccessible methods. I am
            ' leaving it in so it doesn't regress further, but it should be
            ' better.

            Assert.Null(semanticInfo.Type)
            Assert.Null(semanticInfo.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Null(semanticInfo.Symbol)
            Assert.Equal(CandidateReason.Inaccessible, semanticInfo.CandidateReason)
            Assert.Equal(1, semanticInfo.CandidateSymbols.Length)
            Dim sortedCandidates = semanticInfo.CandidateSymbols.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Function Class1.f(a As System.Int32, b As System.Int64) As System.String", sortedCandidates(0).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(0).Kind)

            Assert.Equal(3, semanticInfo.MemberGroup.Length)
            Dim sortedMethodGroup = semanticInfo.MemberGroup.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Function Class1.f(a As System.Int32, b As System.Int64) As System.String", sortedMethodGroup(0).ToTestDisplayString())
            Assert.Equal("Sub Class1.f()", sortedMethodGroup(1).ToTestDisplayString())
            Assert.Equal("Sub Class1.f(x As System.Int32)", sortedMethodGroup(2).ToTestDisplayString())

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub InaccessibleMethodGroup_Constructors_ObjectCreationExpressionSyntax()
            Dim compilation = CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb"><![CDATA[
Imports System

Module Program
    Sub Main(args As String())
        Dim x As New Class1(3, 7)'BIND:"New Class1(3, 7)"
    End Sub

End Module

Class Class1
    Protected Sub New()
    End Sub

    Protected Sub New(x As Integer)
    End Sub

    Private Sub New(ByVal a As Integer, b As Long)
    End Sub
End Class
    ]]></file>
    </compilation>)

            Dim semanticSummary = CompilationUtils.GetSemanticInfoSummary(Of ObjectCreationExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("Class1", semanticSummary.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticSummary.Type.TypeKind)
            Assert.Equal("Class1", semanticSummary.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticSummary.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticSummary.ImplicitConversion.Kind)

            Assert.Null(semanticSummary.Symbol)
            Assert.Equal(CandidateReason.Inaccessible, semanticSummary.CandidateReason)
            Assert.Equal(1, semanticSummary.CandidateSymbols.Length)
            Assert.Equal("Sub Class1..ctor(a As System.Int32, b As System.Int64)", semanticSummary.CandidateSymbols(0).ToTestDisplayString())

            Assert.Null(semanticSummary.Alias)

            Assert.Equal(3, semanticSummary.MemberGroup.Length)
            Dim sortedMethodGroup = semanticSummary.MemberGroup.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Sub Class1..ctor()", sortedMethodGroup(0).ToTestDisplayString())
            Assert.Equal("Sub Class1..ctor(a As System.Int32, b As System.Int64)", sortedMethodGroup(1).ToTestDisplayString())
            Assert.Equal("Sub Class1..ctor(x As System.Int32)", sortedMethodGroup(2).ToTestDisplayString())

            Assert.False(semanticSummary.ConstantValue.HasValue)

        End Sub

        <Fact>
        Public Sub InaccessibleMethodGroup_Constructors_IdentifierNameSyntax()
            Dim compilation = CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb"><![CDATA[
Imports System

Module Program
    Sub Main(args As String())
        Dim x As New Class1(3, 7)'BIND:"Class1"
    End Sub

End Module

Class Class1
    Protected Sub New()
    End Sub

    Protected Sub New(x As Integer)
    End Sub

    Private Sub New(ByVal a As Integer, b As Long)
    End Sub
End Class
    ]]></file>
    </compilation>)

            Dim semanticSummary = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Null(semanticSummary.Type)
            Assert.Null(semanticSummary.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticSummary.ImplicitConversion.Kind)

            Assert.Equal("Class1", semanticSummary.Symbol.ToTestDisplayString())
            Assert.Equal(CandidateReason.None, semanticSummary.CandidateReason)
            Assert.Equal(0, semanticSummary.CandidateSymbols.Length)

            Assert.Null(semanticSummary.Alias)

            Assert.Equal(0, semanticSummary.MemberGroup.Length)

            Assert.False(semanticSummary.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub InaccessibleMethodGroup_AttributeSyntax()
            Dim compilation = CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb"><![CDATA[
Imports System

Module Program

    <Class1(3, 7)>'BIND:"Class1(3, 7)"
    Sub Main(args As String())
    End Sub

End Module

Class Class1
    Inherits Attribute
    Protected Sub New()
    End Sub

    Protected Sub New(x As Integer)
    End Sub

    Private Sub New(ByVal a As Integer, b As Long)
    End Sub
End Class
    ]]></file>
    </compilation>)

            Dim semanticSummary = CompilationUtils.GetSemanticInfoSummary(Of AttributeSyntax)(compilation, "a.vb")

            Assert.Equal("Class1", semanticSummary.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.[Class], semanticSummary.Type.TypeKind)
            Assert.Equal("Class1", semanticSummary.ConvertedType.ToTestDisplayString())
            Assert.Equal(ConversionKind.Identity, semanticSummary.ImplicitConversion.Kind)

            Assert.Null(semanticSummary.Symbol)
            Assert.Equal(CandidateReason.Inaccessible, semanticSummary.CandidateReason)
            Assert.Equal(3, semanticSummary.CandidateSymbols.Length)
            Dim sortedCandidates = semanticSummary.CandidateSymbols.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Sub Class1..ctor()", sortedCandidates(0).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(0).Kind)
            Assert.Equal("Sub Class1..ctor(a As System.Int32, b As System.Int64)", sortedCandidates(1).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(1).Kind)
            Assert.Equal("Sub Class1..ctor(x As System.Int32)", sortedCandidates(2).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(2).Kind)

            Assert.Null(semanticSummary.Alias)

            Assert.Equal(3, semanticSummary.MemberGroup.Length)
            Dim sortedMethodGroup = semanticSummary.MemberGroup.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Sub Class1..ctor()", sortedMethodGroup(0).ToTestDisplayString())
            Assert.Equal("Sub Class1..ctor(a As System.Int32, b As System.Int64)", sortedMethodGroup(1).ToTestDisplayString())
            Assert.Equal("Sub Class1..ctor(x As System.Int32)", sortedMethodGroup(2).ToTestDisplayString())

            Assert.False(semanticSummary.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub InaccessibleMethodGroup_Attribute_IdentifierNameSyntax()
            Dim compilation = CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb"><![CDATA[
Imports System

Module Program

    <Class1(3, 7)>'BIND:"Class1"
    Sub Main(args As String())
    End Sub

End Module

Class Class1
    Inherits Attribute
    Protected Sub New()
    End Sub

    Protected Sub New(x As Integer)
    End Sub

    Private Sub New(ByVal a As Integer, b As Long)
    End Sub
End Class
    ]]></file>
    </compilation>)

            Dim semanticSummary = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("Class1", semanticSummary.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticSummary.Type.TypeKind)
            Assert.Equal("Class1", semanticSummary.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticSummary.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticSummary.ImplicitConversion.Kind)

            Assert.Null(semanticSummary.Symbol)
            Assert.Equal(CandidateReason.Inaccessible, semanticSummary.CandidateReason)
            Assert.Equal(3, semanticSummary.CandidateSymbols.Length)
            Dim sortedCandidates = semanticSummary.CandidateSymbols.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Sub Class1..ctor()", sortedCandidates(0).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(0).Kind)
            Assert.Equal("Sub Class1..ctor(a As System.Int32, b As System.Int64)", sortedCandidates(1).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(1).Kind)
            Assert.Equal("Sub Class1..ctor(x As System.Int32)", sortedCandidates(2).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(2).Kind)

            Assert.Null(semanticSummary.Alias)

            Assert.Equal(3, semanticSummary.MemberGroup.Length)
            Dim sortedMethodGroup = semanticSummary.MemberGroup.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Sub Class1..ctor()", sortedMethodGroup(0).ToTestDisplayString())
            Assert.Equal("Sub Class1..ctor(a As System.Int32, b As System.Int64)", sortedMethodGroup(1).ToTestDisplayString())
            Assert.Equal("Sub Class1..ctor(x As System.Int32)", sortedMethodGroup(2).ToTestDisplayString())

            Assert.False(semanticSummary.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub InaccessibleConstructorsFiltered_ObjectCreationExpressionSyntax()
            Dim compilation = CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb"><![CDATA[
Imports System

Module Program
    Sub Main(args As String())
        Dim x As New Class1(3, 7)'BIND:"New Class1(3, 7)"
    End Sub

End Module

Class Class1
    Protected Sub New()
    End Sub

    Public Sub New(x As Integer)
    End Sub

    Public Sub New(ByVal a As Integer, b As Long)
    End Sub
End Class
    ]]></file>
    </compilation>)

            Dim semanticSummary = CompilationUtils.GetSemanticInfoSummary(Of ObjectCreationExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("Class1", semanticSummary.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticSummary.Type.TypeKind)
            Assert.Equal("Class1", semanticSummary.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticSummary.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticSummary.ImplicitConversion.Kind)

            Assert.Equal("Sub Class1..ctor(a As System.Int32, b As System.Int64)", semanticSummary.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, semanticSummary.Symbol.Kind)
            Assert.Equal(0, semanticSummary.CandidateSymbols.Length)

            Assert.Null(semanticSummary.Alias)

            Assert.Equal(2, semanticSummary.MemberGroup.Length)
            Dim sortedMethodGroup = semanticSummary.MemberGroup.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Sub Class1..ctor(a As System.Int32, b As System.Int64)", sortedMethodGroup(0).ToTestDisplayString())
            Assert.Equal("Sub Class1..ctor(x As System.Int32)", sortedMethodGroup(1).ToTestDisplayString())

            Assert.False(semanticSummary.ConstantValue.HasValue)

        End Sub

        <Fact>
        Public Sub InaccessibleConstructorsFiltered_IdentifierNameSyntax()
            Dim compilation = CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb"><![CDATA[
Imports System

Module Program
    Sub Main(args As String())
        Dim x As New Class1(3, 7)'BIND:"Class1"
    End Sub

End Module

Class Class1
    Protected Sub New()
    End Sub

    Public Sub New(x As Integer)
    End Sub

    Public Sub New(ByVal a As Integer, b As Long)
    End Sub
End Class
    ]]></file>
    </compilation>)

            Dim semanticSummary = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Null(semanticSummary.Type)
            Assert.Null(semanticSummary.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticSummary.ImplicitConversion.Kind)

            Assert.Equal("Class1", semanticSummary.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.NamedType, semanticSummary.Symbol.Kind)
            Assert.Equal(0, semanticSummary.CandidateSymbols.Length)

            Assert.Null(semanticSummary.Alias)

            Assert.Equal(0, semanticSummary.MemberGroup.Length)

            Assert.False(semanticSummary.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub InaccessibleConstructorsFiltered_IdentifierNameSyntax2()
            Dim compilation = CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb"><![CDATA[
Imports System

Module Program
    Sub Main(args As String())
        Dim x As New Class1(3, 7)'BIND:"New Class1(3, 7)"
    End Sub

End Module

Class Class1
    Protected Sub New()
    End Sub

    Public Sub New(x As Integer)
    End Sub

    Public Sub New(ByVal a As Integer, b As Long)
    End Sub
End Class
    ]]></file>
    </compilation>)

            Dim semanticSummary = CompilationUtils.GetSemanticInfoSummary(Of ObjectCreationExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("Class1", semanticSummary.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticSummary.Type.TypeKind)
            Assert.Equal("Class1", semanticSummary.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticSummary.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticSummary.ImplicitConversion.Kind)

            Assert.Equal("Sub Class1..ctor(a As System.Int32, b As System.Int64)", semanticSummary.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, semanticSummary.Symbol.Kind)
            Assert.Equal(0, semanticSummary.CandidateSymbols.Length)

            Assert.Null(semanticSummary.Alias)

            Assert.Equal(2, semanticSummary.MemberGroup.Length)
            Dim sortedMethodGroup = semanticSummary.MemberGroup.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Sub Class1..ctor(a As System.Int32, b As System.Int64)", sortedMethodGroup(0).ToTestDisplayString())
            Assert.Equal("Sub Class1..ctor(x As System.Int32)", sortedMethodGroup(1).ToTestDisplayString())

            Assert.False(semanticSummary.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub InaccessibleConstructorsFiltered_AttributeSyntax()
            Dim compilation = CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb"><![CDATA[
Imports System

Module Program

    <Class1(3, 7)>'BIND:"Class1(3, 7)"
    Sub Main(args As String())
    End Sub

End Module

Class Class1
    Inherits Attribute
    Protected Sub New()
    End Sub

    Public Sub New(x As Integer)
    End Sub

    Public Sub New(ByVal a As Integer, b As Long)
    End Sub
End Class
    ]]></file>
    </compilation>)

            Dim semanticSummary = CompilationUtils.GetSemanticInfoSummary(Of AttributeSyntax)(compilation, "a.vb")

            Assert.Equal("Class1", semanticSummary.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticSummary.Type.TypeKind)
            Assert.Equal("Class1", semanticSummary.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticSummary.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticSummary.ImplicitConversion.Kind)

            Assert.Equal("Sub Class1..ctor(a As System.Int32, b As System.Int64)", semanticSummary.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, semanticSummary.Symbol.Kind)
            Assert.Equal(0, semanticSummary.CandidateSymbols.Length)

            Assert.Null(semanticSummary.Alias)

            Assert.Equal(2, semanticSummary.MemberGroup.Length)
            Dim sortedMethodGroup = semanticSummary.MemberGroup.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Sub Class1..ctor(a As System.Int32, b As System.Int64)", sortedMethodGroup(0).ToTestDisplayString())
            Assert.Equal("Sub Class1..ctor(x As System.Int32)", sortedMethodGroup(1).ToTestDisplayString())

            Assert.False(semanticSummary.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub InaccessibleConstructorsFiltered_Attribute_IdentifierNameSyntax()
            Dim compilation = CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb"><![CDATA[
Imports System

Module Program

    <Class1(3, 7)>'BIND:"Class1"
    Sub Main(args As String())
    End Sub

End Module

Class Class1
    Inherits Attribute
    Protected Sub New()
    End Sub

    Public Sub New(x As Integer)
    End Sub

    Public Sub New(ByVal a As Integer, b As Long)
    End Sub
End Class
    ]]></file>
    </compilation>)

            Dim semanticSummary = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("Class1", semanticSummary.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticSummary.Type.TypeKind)
            Assert.Equal("Class1", semanticSummary.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticSummary.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticSummary.ImplicitConversion.Kind)

            Assert.Equal("Sub Class1..ctor(a As System.Int32, b As System.Int64)", semanticSummary.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, semanticSummary.Symbol.Kind)
            Assert.Equal(0, semanticSummary.CandidateSymbols.Length)

            Assert.Null(semanticSummary.Alias)

            Assert.Equal(2, semanticSummary.MemberGroup.Length)
            Dim sortedMethodGroup = semanticSummary.MemberGroup.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Sub Class1..ctor(a As System.Int32, b As System.Int64)", sortedMethodGroup(0).ToTestDisplayString())
            Assert.Equal("Sub Class1..ctor(x As System.Int32)", sortedMethodGroup(1).ToTestDisplayString())

            Assert.False(semanticSummary.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub Invocation1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System

Module Program
    Sub Main(args As String())
        f(3, 7)'BIND:"f(3, 7)"
    End Sub

    Sub f()
    End Sub

    Sub f(x As Integer)
    End Sub

    Function f(ByVal a As Integer, b As Long) As String
        Return "hi"
    End Function
End Module
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of InvocationExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("System.String", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("System.String", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Function Program.f(a As System.Int32, b As System.Int64) As System.String", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub Invocation2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System

Module Program
    Sub Main(args As String())
        Dim x As New Class1()
        x.f(3, 7)'BIND:"x.f(3, 7)"
    End Sub

End Module

Class Class1
    Public Sub f()
    End Sub

    Public Sub f(x As Integer)
    End Sub

    Public Function f(ByVal a As Integer, b As Long) As String
        Return "hi"
    End Function
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of InvocationExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("System.String", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("System.String", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Function Class1.f(a As System.Int32, b As System.Int64) As System.String", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub InaccessibleInvocation()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System

Module Program
    Sub Main(args As String())
        Dim x As New Class1()
        x.f(3, 7)'BIND:"x.f(3, 7)"
    End Sub

End Module

Class Class1
    Protected Sub f()
    End Sub

    Protected Sub f(x As Integer)
    End Sub

    Private Function f(ByVal a As Integer, b As Long) As String
        Return "hi"
    End Function
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of InvocationExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("System.String", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("System.String", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Null(semanticInfo.Symbol)
            Assert.Equal(CandidateReason.Inaccessible, semanticInfo.CandidateReason)
            Assert.Equal(1, semanticInfo.CandidateSymbols.Length)
            Dim sortedCandidates = semanticInfo.CandidateSymbols.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Function Class1.f(a As System.Int32, b As System.Int64) As System.String", sortedCandidates(0).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(0).Kind)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub Property1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System
Imports System.Collections.Generic
Imports System.Linq

Module Program
    Sub Main(args As String())
        Dim a As New X
        a.Prop = 4'BIND:"Prop"
    End Sub
End Module

Class X
    Public Property Prop As String
        Get
            Return ""
        End Get
        Set
        End Set
    End Property
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("System.String", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("System.String", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Property X.Prop As System.String", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Property, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub Property2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System
Imports System.Collections.Generic
Imports System.Linq

Module Program
    Sub Main(args As String())
        Dim a As New X
        Dim u As String = a.Prop'BIND:"a.Prop"
    End Sub
End Module

Class X
    Public Property Prop As String
        Get
            Return ""
        End Get
        Set
        End Set
    End Property
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of MemberAccessExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("System.String", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("System.String", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Property X.Prop As System.String", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Property, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub SimpleConstantExpression()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System
Imports System.Collections.Generic
Imports System.Linq

Module Program
    Sub Main(args As String())
        Dim i As Integer = 10 * 10'BIND:"10 * 10"
    End Sub
End Module

Class X
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of BinaryExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("System.Int32", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Int32", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Function System.Int32.op_CheckedMultiply(left As System.Int32, right As System.Int32) As System.Int32",
                         semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(CandidateReason.None, semanticInfo.CandidateReason)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.True(semanticInfo.ConstantValue.HasValue)
            Assert.Equal(100, semanticInfo.ConstantValue.Value)
        End Sub

        <Fact>
        Public Sub InaccessibleParameter()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System
Imports System.Collections.Generic
Imports System.Linq

Module Program
    Sub goo(a As Outer.Inner)
        Dim q As Integer
        q = a.x'BIND:"a"
    End Sub
End Module

Class Outer
    Private Class Inner
        Public x As Integer
    End Class
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("Outer.Inner", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("Outer.Inner", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("a As Outer.Inner", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Parameter, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)

            Dim paramSym As ParameterSymbol = DirectCast(semanticInfo.Symbol, ParameterSymbol)
            Assert.Equal(TypeKind.Error, paramSym.Type.TypeKind)
        End Sub

        <WorkItem(538447, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/538447")>
        <Fact>
        Public Sub CastInterfaceToArray()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Module Module1
    Sub Main()
        Dim i As ICloneable = New Integer() {}
        Dim arr() As Integer
        arr = CType(i, Integer())'BIND:"CType(i, Integer())"
        arr = DirectCast(i, Integer())
        arr = TryCast(i, Integer())
    End Sub
End Module
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of CastExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("System.Int32()", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Array, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Int32()", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Array, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Null(semanticInfo.Symbol)
            Assert.Equal(CandidateReason.None, semanticInfo.CandidateReason)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub ImplementsClause1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Interface IA(Of T)
    Overloads Sub Goo(ByVal x As T)
    Overloads Sub Goo(ByVal x As String)
End Interface

Interface IC(Of T)
    Inherits IA(Of T)
End Interface

Interface IB
    Overloads Sub Bar(x As String)
    ReadOnly Property P As Long
    Property R(x As String) As Long
End Interface

Class K
    Implements IC(Of Integer)
    Implements IB

    Public Overloads Sub F(x As String) Implements IB.Bar, IC(Of Integer).Goo 'BIND1:"IB.Bar" 'BIND2:"Goo"
    End Sub

    Public Overloads Sub F(x As Integer) Implements IA(Of Integer).Goo  'BIND3:"IA(Of Integer).Goo" 
    End Sub

    Public Property Q(x As String) As Long Implements IB.R 'BIND4:"IB.R"
        Get
            Return 0
        End Get
        Set
        End Set
    End Property

    Public Property Q As Long Implements IB.P 'BIND5:"IB"
        Get
            Return 0
        End Get
        Set
        End Set
    End Property
End Class

    </file>
</compilation>)
            'IB.Bar
            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of QualifiedNameSyntax)(compilation, "a.vb", 1)

            Assert.Null(semanticInfo.Type)
            Assert.Null(semanticInfo.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Sub IB.Bar(x As System.String)", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)

            'Goo
            semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb", 2)

            Assert.Null(semanticInfo.Type)
            Assert.Null(semanticInfo.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Sub IA(Of System.Int32).Goo(x As System.String)", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)

            ' IA(Of Integer).Goo
            semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of QualifiedNameSyntax)(compilation, "a.vb", 3)

            Assert.Null(semanticInfo.Type)
            Assert.Null(semanticInfo.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Sub IA(Of System.Int32).Goo(x As System.Int32)", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)

            ' IB.R
            semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of QualifiedNameSyntax)(compilation, "a.vb", 4)

            Assert.Null(semanticInfo.Type)
            Assert.Null(semanticInfo.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Property IB.R(x As System.String) As System.Int64", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Property, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)

            ' IB
            semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb", 5)

            Assert.Equal("IB", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Interface, semanticInfo.Type.TypeKind)
            Assert.Equal("IB", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Interface, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("IB", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.NamedType, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub ImplementsClause2()
            Dim compilation = CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Option Strict On
Imports System

Interface I1
    Sub goo(x As Integer)
End Interface

Class C1
    Implements I1

    Public Function goo(x As Integer) As String Implements I1.goo'BIND:"I1.goo"
        Throw New NotImplementedException()
    End Function
End Class

    ]]></file>
</compilation>)

            Dim semanticSummary = CompilationUtils.GetSemanticInfoSummary(Of QualifiedNameSyntax)(compilation, "a.vb")

            Assert.Null(semanticSummary.Type)
            Assert.Null(semanticSummary.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticSummary.ImplicitConversion.Kind)

            Assert.Null(semanticSummary.Symbol)
            Assert.Equal(CandidateReason.OverloadResolutionFailure, semanticSummary.CandidateReason)
            Assert.Equal(1, semanticSummary.CandidateSymbols.Length)
            Dim sortedCandidates = semanticSummary.CandidateSymbols.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Sub I1.goo(x As System.Int32)", sortedCandidates(0).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(0).Kind)

            Assert.Null(semanticSummary.Alias)

            Assert.Equal(0, semanticSummary.MemberGroup.Length)

            Assert.False(semanticSummary.ConstantValue.HasValue)
        End Sub

        <ConditionalFact(GetType(WindowsOnly), Reason:="https://github.com/dotnet/roslyn/issues/29531")>
        Public Sub ImplementsClause3()
            Dim compilation = CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Option Strict On
Imports System

Interface I1
    Sub goo(x As Integer)
    Sub goo(x As Integer, y As Integer)
End Interface

Class C1
    Implements I1

    Public Sub goo(x As Long) Implements I1.goo'BIND:"I1.goo"
        Throw New NotImplementedException()
    End Sub
End Class

    ]]></file>
</compilation>)

            Dim semanticSummary = CompilationUtils.GetSemanticInfoSummary(Of QualifiedNameSyntax)(compilation, "a.vb")

            Assert.Null(semanticSummary.Type)
            Assert.Null(semanticSummary.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticSummary.ImplicitConversion.Kind)

            Assert.Null(semanticSummary.Symbol)
            Assert.Equal(CandidateReason.OverloadResolutionFailure, semanticSummary.CandidateReason)
            Assert.Equal(2, semanticSummary.CandidateSymbols.Length)
            Dim sortedCandidates = semanticSummary.CandidateSymbols.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString(), StringComparer.Ordinal).ToArray()
            Assert.Equal("Sub I1.goo(x As System.Int32)", sortedCandidates(0).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(0).Kind)
            Assert.Equal("Sub I1.goo(x As System.Int32, y As System.Int32)", sortedCandidates(1).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(1).Kind)

            Assert.Null(semanticSummary.Alias)

            Assert.Equal(0, semanticSummary.MemberGroup.Length)

            Assert.False(semanticSummary.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub ImplementsClause4()
            Dim compilation = CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Option Strict On
Imports System

Interface I1
    Sub goo(x As Integer)
    Sub goo(x As Integer, y As Integer)
End Interface

Class C1
    Public Sub goo(x As Integer) Implements I1.goo 'BIND:"I1.goo"'BIND:"I1.goo"
        Throw New NotImplementedException()
    End Sub
End Class
    ]]></file>
</compilation>)

            Dim semanticSummary = CompilationUtils.GetSemanticInfoSummary(Of QualifiedNameSyntax)(compilation, "a.vb")

            Assert.Null(semanticSummary.Type)
            Assert.Null(semanticSummary.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticSummary.ImplicitConversion.Kind)

            Assert.Null(semanticSummary.Symbol)
            Assert.Equal(CandidateReason.NotReferencable, semanticSummary.CandidateReason)
            Assert.Equal(1, semanticSummary.CandidateSymbols.Length)
            Dim sortedCandidates = semanticSummary.CandidateSymbols.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Sub I1.goo(x As System.Int32)", sortedCandidates(0).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(0).Kind)

            Assert.Null(semanticSummary.Alias)

            Assert.Equal(0, semanticSummary.MemberGroup.Length)

            Assert.False(semanticSummary.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub ImplementsClause5()
            Dim compilation = CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Option Strict On
Imports System

Interface I1
    Private Sub goo(x As Integer)
End Interface

Class C1
    Implements I1
    Public Sub goo(x As Integer) Implements I1.goo 'BIND:"I1.goo"'BIND:"I1.goo"
        Throw New NotImplementedException()
    End Sub
End Class
    ]]></file>
</compilation>)

            Dim semanticSummary = CompilationUtils.GetSemanticInfoSummary(Of QualifiedNameSyntax)(compilation, "a.vb")

            Assert.Null(semanticSummary.Type)
            Assert.Null(semanticSummary.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticSummary.ImplicitConversion.Kind)

            Assert.Null(semanticSummary.Symbol)
            Assert.Equal(CandidateReason.Inaccessible, semanticSummary.CandidateReason)
            Assert.Equal(1, semanticSummary.CandidateSymbols.Length)
            Dim sortedCandidates = semanticSummary.CandidateSymbols.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Sub I1.goo(x As System.Int32)", sortedCandidates(0).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(0).Kind)

            Assert.Null(semanticSummary.Alias)

            Assert.Equal(0, semanticSummary.MemberGroup.Length)

            Assert.False(semanticSummary.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub ImplementsClause6()
            Dim compilation = CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Option Strict On
Imports System

Interface I1
    Inherits I2, I3
End Interface

Interface I2
    Sub goo(x As Integer, z As String)
End Interface

Interface I3
    Sub goo(x As Integer, y As Integer)
End Interface

Class C1
    Public Sub goo(x As Integer) Implements I1.goo 'BIND:"I1.goo"'BIND:"I1.goo"
        Throw New NotImplementedException()
    End Sub
End Class
    ]]></file>
</compilation>)

            Dim semanticSummary = CompilationUtils.GetSemanticInfoSummary(Of QualifiedNameSyntax)(compilation, "a.vb")

            Assert.Null(semanticSummary.Type)
            Assert.Null(semanticSummary.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticSummary.ImplicitConversion.Kind)

            Assert.Null(semanticSummary.Symbol)
            Assert.Equal(CandidateReason.OverloadResolutionFailure, semanticSummary.CandidateReason)
            Assert.Equal(2, semanticSummary.CandidateSymbols.Length)
            Dim sortedCandidates = semanticSummary.CandidateSymbols.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Sub I2.goo(x As System.Int32, z As System.String)", sortedCandidates(0).ToTestDisplayString())
            Assert.Equal("Sub I3.goo(x As System.Int32, y As System.Int32)", sortedCandidates(1).ToTestDisplayString())

            Assert.Null(semanticSummary.Alias)

            Assert.Equal(0, semanticSummary.MemberGroup.Length)

            Assert.False(semanticSummary.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub ImplementsClause7()
            Dim compilation = CreateCompilationWithMscorlib40(
            <compilation>
                <file name="a.vb"><![CDATA[
Option Strict On
Imports System

Module m1
    Public Sub main()

    End Sub
End Module

Interface I1
    Inherits I2, I3
End Interface

Interface I2
    Event E1()
End Interface

Interface I3
    Event E2 As I1.E1EventHandler
End Interface

Class C1
    Implements I1

    Public Event E3() Implements I2.E1, I3.E2'BIND:"I3.E2"
End Class
    ]]></file>
            </compilation>)

            Dim semanticSummary = CompilationUtils.GetSemanticInfoSummary(Of QualifiedNameSyntax)(compilation, "a.vb")

            Assert.Null(semanticSummary.Type)
            Assert.Null(semanticSummary.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticSummary.ImplicitConversion.Kind)

            Assert.Equal("Event I3.E2 As I2.E1EventHandler", semanticSummary.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Event, semanticSummary.Symbol.Kind)
            Assert.Equal(0, semanticSummary.CandidateSymbols.Length)

            Assert.Null(semanticSummary.Alias)

            Assert.Equal(0, semanticSummary.MemberGroup.Length)

            Assert.False(semanticSummary.ConstantValue.HasValue)
        End Sub

        <Fact()>
        Public Sub InterfaceImplementationCantFindMatching()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Interface IA
    Overloads Sub Goo(ByVal x As Long)
    Overloads Sub Goo(ByVal x As String)
End Interface

Class K
    Implements IA

    Public Overloads Sub F(x As Integer) Implements IA.Goo'BIND:"IA.Goo"
    End Sub
End Class

    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of QualifiedNameSyntax)(compilation, "a.vb")

            Assert.Null(semanticInfo.Type)
            Assert.Null(semanticInfo.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Null(semanticInfo.Symbol)
            Assert.True(semanticInfo.CandidateSymbols.Length > 0, "Should have candidate symbols")
            Assert.Equal(CandidateReason.OverloadResolutionFailure, semanticInfo.CandidateReason)
            Assert.False(semanticInfo.MemberGroup.Length > 0, "Shouldn't have member group")

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <WorkItem(539111, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/539111")>
        <Fact>
        Public Sub MethodReferenceWithImplicitTypeArguments()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System
Module Module1
    Sub Main()
        Goo(Of Integer)'BIND:"Goo(Of Integer)"
    End Sub

    Sub Goo(a As String)
    End Sub

    Sub Goo(ByRef a As String, b As String)
    End Sub

    Sub Goo(Of T)(a As String, b As String)
    End Sub
End Module
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of GenericNameSyntax)(compilation, "a.vb")

            Assert.Null(semanticInfo.Type)
            Assert.Null(semanticInfo.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Null(semanticInfo.Symbol)
            Assert.Equal(CandidateReason.OverloadResolutionFailure, semanticInfo.CandidateReason)
            Assert.Equal(1, semanticInfo.CandidateSymbols.Length)
            Dim sortedCandidates = semanticInfo.CandidateSymbols.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Sub Module1.Goo(Of System.Int32)(a As System.String, b As System.String)", sortedCandidates(0).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(0).Kind)

            Assert.Equal(1, semanticInfo.MemberGroup.Length)
            Dim sortedMethodGroup = semanticInfo.MemberGroup.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Sub Module1.Goo(Of System.Int32)(a As System.String, b As System.String)", sortedMethodGroup(0).ToTestDisplayString())

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <WorkItem(538452, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/538452")>
        <Fact>
        Public Sub InvalidMethodInvocationExpr()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Sub Main()
    T()'BIND:"T()"
End Sub

Sub T()
End Sub
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of InvocationExpressionSyntax)(compilation, "a.vb")

            Assert.Equal(SpecialType.System_Void, semanticInfo.Type.SpecialType)
            Assert.Equal(SpecialType.System_Void, semanticInfo.ConvertedType.SpecialType)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("T", semanticInfo.Symbol.Name)
            Assert.Equal(CandidateReason.None, semanticInfo.CandidateReason)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub UnaryPlusExprWithoutMsCorlibRef()
            Dim compilation = CreateEmptyCompilationWithReferences(
<compilation>
    <file name="a.vb">
Module Module1
    Sub Main()
        Dim count As Integer
        count = +10 'BIND:"+10"'BIND:"+10"
    End Sub
End Module
    </file>
</compilation>, {})

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of UnaryExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("System.Int32[missing]", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Error, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Int32[missing]", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Error, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Function System.Int32[missing].op_UnaryPlus(value As System.Int32[missing]) As System.Int32[missing]",
                         semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(CandidateReason.None, semanticInfo.CandidateReason)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <WorkItem(4280, "DevDiv_Projects/Roslyn")>
        <Fact()>
        Public Sub BindingIsNothingFunc()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation>
    <file name="a.vb">
Imports Microsoft.VisualBasic
Module Module1
    Sub Main()
        Dim var1 As Object
        If IsNothing(var1) Then'BIND:"IsNothing(var1)"
        End If
    End Sub
End Module
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of InvocationExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("System.Boolean", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.Type.TypeKind)
        End Sub

        <Fact>
        Public Sub MaxIntPlusOneHexLiteralConst()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Module Module1
    Sub Main()
        Dim lngPass As Integer
        lngPass = &amp;H80000000'BIND:"&amp;H80000000"
    End Sub
End Module
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of LiteralExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("System.Int32", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Int32", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Null(semanticInfo.Symbol)
            Assert.Equal(CandidateReason.None, semanticInfo.CandidateReason)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.True(semanticInfo.ConstantValue.HasValue)
            Assert.Equal(CInt(-2147483648), semanticInfo.ConstantValue.Value)
        End Sub

        <WorkItem(539017, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/539017")>
        <Fact>
        Public Sub ParenExprInMultiDimArrayDeclWithError()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Module Mod1
    Sub Main()
        Dim scen3(, 5,6,) As Integer
        Dim x((,)) As Integer 'BIND:"(,)"
    End Sub
End Module
    ]]></file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of TupleExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("(?, ?)", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Struct, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Int32", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(Nothing, semanticInfo.ImplicitConversion.Kind)

            Assert.Null(semanticInfo.Symbol)
            Assert.Equal(CandidateReason.None, semanticInfo.CandidateReason)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub InvocExprWithImplicitlyTypedArgument()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Module Module1
    Sub VerifyByteArray(ByRef arry() As Byte, ByRef lbnd As Integer)
    End Sub

    Sub Main()
        Dim vsarry() As Byte
        Dim Max = 140000
        VerifyByteArray(vsarry, Max)'BIND:"VerifyByteArray(vsarry, Max)"
    End Sub
End Module
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of InvocationExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("System.Void", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Void", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Sub Module1.VerifyByteArray(ByRef arry As System.Byte(), ByRef lbnd As System.Int32)", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <WorkItem(4512, "DevDiv_Projects/Roslyn")>
        <Fact>
        Public Sub MultiDimArrayCreationExpr()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Module Module1
    Sub Main()
        Dim y = New Object() {1, 2}
        Dim x As Object = New Object()() {y, y}'BIND:"New Object()() {y, y}"
    End Sub
End Module
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of ArrayCreationExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("System.Object()()", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Array, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Object", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.WideningReference, semanticInfo.ImplicitConversion.Kind)

            Assert.Null(semanticInfo.Symbol)
            Assert.Equal(CandidateReason.None, semanticInfo.CandidateReason)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <WorkItem(527716, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/527716")>
        <Fact>
        Public Sub EmptyParenExprInArrayDeclWithError()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Module Mod1
    Sub Main()
        Dim x1 = New Single(0, (* ) - 1) {{2}, {4}}'BIND:"(* )"
    End Sub
End Module
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of ParenthesizedExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("?", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Error, semanticInfo.Type.TypeKind)
            Assert.Equal("?", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Error, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Null(semanticInfo.Symbol)
            Assert.Equal(CandidateReason.None, semanticInfo.CandidateReason)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <WorkItem(538918, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/538918")>
        <Fact>
        Public Sub MeSymbol()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Class C
    Sub Test()
        Me.Test()'BIND:"Me"
    End Sub
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of MeExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("C", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("C", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Me As C", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Parameter, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)
            Assert.True(DirectCast(semanticInfo.Symbol, ParameterSymbol).IsMe, "should be Me symbol")

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <WorkItem(527818, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/527818")>
        <Fact>
        Public Sub BindingFuncNoBracket()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System
Imports Microsoft.VisualBasic

Namespace VBNS
    Class Test

        Function MyFunc() As Byte
            Return Nothing
        End Function

        Sub MySub()
            Dim ret As Byte = MyFunc'BIND:"MyFunc"
        End Sub
    End Class
End Namespace
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("System.Byte", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Byte", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Function VBNS.Test.MyFunc() As System.Byte", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub NamespaceAlias()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports S = System

Namespace NS
    Class Test
        Dim x As S.Exception'BIND:"S"
    End Class
End Namespace
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Null(semanticInfo.Type)
            Assert.Null(semanticInfo.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.NotNull(semanticInfo.Symbol)
            Assert.NotNull(semanticInfo.Alias)
            Assert.Equal("S=System", semanticInfo.Alias.ToTestDisplayString())
            Assert.Equal(SymbolKind.Alias, semanticInfo.Alias.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Dim ns = DirectCast(semanticInfo.Alias.Target, NamespaceSymbol)
            Assert.Equal(ns.Name, "System")

            Assert.Equal(compilation.SourceModule, semanticInfo.Alias.ContainingModule)
            Assert.Equal(compilation.Assembly, semanticInfo.Alias.ContainingAssembly)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub NamespaceAlias2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System
Imports N = NS1.NS2

Namespace NS1.NS2
    Public Class A
        Public Shared Sub M
            Dim o As N.A'BIND:"N"

        End Sub
    End Class
End Namespace

    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Null(semanticInfo.Type)
            Assert.Null(semanticInfo.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.NotNull(semanticInfo.Symbol)
            Assert.NotNull(semanticInfo.Alias)
            Assert.Equal("N=NS1.NS2", semanticInfo.Alias.ToTestDisplayString())
            Assert.Equal(SymbolKind.Alias, semanticInfo.Alias.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Dim ns = DirectCast(semanticInfo.Alias.Target, NamespaceSymbol)
            Assert.Equal("NS1.NS2", ns.ToTestDisplayString())

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub TypeAlias()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports S = System.String

Namespace NS
    Class Test
        Sub Goo
            Dim x As String
            x = S.Empty'BIND:"S"
        End Sub
    End Class
End Namespace
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("System.String", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("System.String", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.NotNull(semanticInfo.Symbol)
            Assert.NotNull(semanticInfo.Alias)
            Assert.Equal("S=System.String", semanticInfo.Alias.ToTestDisplayString())
            Assert.Equal(SymbolKind.Alias, semanticInfo.Alias.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)
            Assert.Equal("String", semanticInfo.Alias.Target.Name)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub TypeAlias2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System
Imports System.Collections.Generic
Imports System.Linq
Imports T = System.Guid

Module Program
    Sub Main(args As String())
        Dim a As Type
        a = GetType(T)'BIND:"T"
    End Sub
End Module
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("System.Guid", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Guid", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.NotNull(semanticInfo.Symbol)
            Assert.NotNull(semanticInfo.Alias)
            Assert.Equal("T=System.Guid", semanticInfo.Alias.ToTestDisplayString())
            Assert.Equal(SymbolKind.Alias, semanticInfo.Alias.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub TypeAlias3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System
Imports System.Collections.Generic
Imports System.Linq
Imports T = System.Guid

Module Program
    Dim a As T'BIND:"T"
    Sub Main(args As String())

    End Sub
End Module
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("System.Guid", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Guid", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.NotNull(semanticInfo.Symbol)
            Assert.NotNull(semanticInfo.Alias)
            Assert.Equal("T=System.Guid", semanticInfo.Alias.ToTestDisplayString())
            Assert.Equal(SymbolKind.Alias, semanticInfo.Alias.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <WorkItem(540279, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/540279")>
        <Fact>
        Public Sub NoMembersForVoidReturnType()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Class C
    Sub Test()
        System.Console.WriteLine()'BIND:"System.Console.WriteLine()"
    End Sub
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of InvocationExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("System.Void", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Void", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Sub System.Console.WriteLine()", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)

            Dim semanticModel = compilation.GetSemanticModel(CompilationUtils.GetTree(compilation, "a.vb"))
            Dim methodSymbol As MethodSymbol = DirectCast(semanticInfo.Symbol, MethodSymbol)
            Dim returnType = methodSymbol.ReturnType
            Dim symbols = semanticModel.LookupSymbols(0, returnType)
            Assert.Equal(0, symbols.Length)
        End Sub

        <Fact>
        Public Sub EnumMember1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Enum filePermissions
    create = 1
    read = create'BIND:"create"
    write = 4
    delete = 8
End Enum
Class c1
    Public Shared Sub Main(args As String())
        Dim file1Perm As filePermissions
        file1Perm = filePermissions.create Or filePermissions.read
    End Sub
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("filePermissions", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Enum, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Int32", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.WideningNumeric Or ConversionKind.InvolvesEnumTypeConversions, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("filePermissions.create", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Field, semanticInfo.Symbol.Kind)
            Assert.True(TypeOf semanticInfo.Symbol Is SourceEnumConstantSymbol, "Should have bound to synthesized enum constant")
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.True(semanticInfo.ConstantValue.HasValue)
            Assert.Equal(1, semanticInfo.ConstantValue.Value)
        End Sub

        <Fact>
        Public Sub CatchVariable()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Class c1
    Public Shared Sub Main()
        Try 
        Catch ex as Exception  'BIND:"ex"
        End Try
    End Sub
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("Exception", semanticInfo.Type.ToTestDisplayString())
        End Sub

        <Fact>
        Public Sub CatchVariable1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Class c1
    Public Shared Sub Main()
        dim a as Action = Sub()
                            Try 
                            Catch ex as Exception  'BIND:"ex"
                            End Try
                          End Sub
    End Sub
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("Exception", semanticInfo.Type.ToTestDisplayString())
        End Sub

        <WorkItem(15925, "DevDiv_Projects/Roslyn")>
        <WorkItem(540050, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/540050")>
        <Fact>
        Public Sub StaticLocalSymbol()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Class C
    Public Function goo() As Integer
        Static i As Integer = 23
        i = i + 1'BIND:"i"
        Return i
    End Function

    Public Shared Sub Main()
    End Sub
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("System.Int32", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Int32", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("i As System.Int32", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Local, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)

            Dim iSymbol = DirectCast(semanticInfo.Symbol, LocalSymbol)
            Assert.True(iSymbol.IsStatic)
            Assert.False(iSymbol.IsShared)
        End Sub

        <ConditionalFact(GetType(WindowsOnly), Reason:="https://github.com/dotnet/roslyn/issues/29531")>
        Public Sub IncompleteWriteLine()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System
Imports System.Collections.Generic
Imports System.Linq

Module Program
    Sub Main(args As String())
        Console.WriteLine( 'BIND:"WriteLine"
    End Sub
End Module
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Null(semanticInfo.Type)
            Assert.Null(semanticInfo.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Null(semanticInfo.Symbol)
            Assert.Equal(CandidateReason.OverloadResolutionFailure, semanticInfo.CandidateReason)
            Assert.Equal(12, semanticInfo.CandidateSymbols.Length)
            Dim sortedCandidates = semanticInfo.CandidateSymbols.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString(), StringComparer.InvariantCulture).ToArray()
            Assert.Equal("Sub System.Console.WriteLine(buffer As System.Char())", sortedCandidates(0).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(0).Kind)
            Assert.Equal("Sub System.Console.WriteLine(value As System.Boolean)", sortedCandidates(1).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(1).Kind)
            Assert.Equal("Sub System.Console.WriteLine(value As System.Char)", sortedCandidates(2).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(2).Kind)
            Assert.Equal("Sub System.Console.WriteLine(value As System.Decimal)", sortedCandidates(3).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(3).Kind)
            Assert.Equal("Sub System.Console.WriteLine(value As System.Double)", sortedCandidates(4).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(4).Kind)
            Assert.Equal("Sub System.Console.WriteLine(value As System.Int32)", sortedCandidates(5).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(5).Kind)
            Assert.Equal("Sub System.Console.WriteLine(value As System.Int64)", sortedCandidates(6).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(6).Kind)
            Assert.Equal("Sub System.Console.WriteLine(value As System.Object)", sortedCandidates(7).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(7).Kind)
            Assert.Equal("Sub System.Console.WriteLine(value As System.Single)", sortedCandidates(8).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(8).Kind)
            Assert.Equal("Sub System.Console.WriteLine(value As System.String)", sortedCandidates(9).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(9).Kind)
            Assert.Equal("Sub System.Console.WriteLine(value As System.UInt32)", sortedCandidates(10).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(10).Kind)
            Assert.Equal("Sub System.Console.WriteLine(value As System.UInt64)", sortedCandidates(11).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(11).Kind)

            Assert.Equal(19, semanticInfo.MemberGroup.Length)
            Dim sortedMethodGroup = semanticInfo.MemberGroup.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString(), StringComparer.Ordinal).ToArray()
            Assert.Equal("Sub System.Console.WriteLine()", sortedMethodGroup(0).ToTestDisplayString())
            Assert.Equal("Sub System.Console.WriteLine(buffer As System.Char())", sortedMethodGroup(1).ToTestDisplayString())
            Assert.Equal("Sub System.Console.WriteLine(buffer As System.Char(), index As System.Int32, count As System.Int32)", sortedMethodGroup(2).ToTestDisplayString())
            Assert.Equal("Sub System.Console.WriteLine(format As System.String, ParamArray arg As System.Object())", sortedMethodGroup(3).ToTestDisplayString())
            Assert.Equal("Sub System.Console.WriteLine(format As System.String, arg0 As System.Object)", sortedMethodGroup(4).ToTestDisplayString())
            Assert.Equal("Sub System.Console.WriteLine(format As System.String, arg0 As System.Object, arg1 As System.Object)", sortedMethodGroup(5).ToTestDisplayString())
            Assert.Equal("Sub System.Console.WriteLine(format As System.String, arg0 As System.Object, arg1 As System.Object, arg2 As System.Object)", sortedMethodGroup(6).ToTestDisplayString())
            Assert.Equal("Sub System.Console.WriteLine(format As System.String, arg0 As System.Object, arg1 As System.Object, arg2 As System.Object, arg3 As System.Object)", sortedMethodGroup(7).ToTestDisplayString())
            Assert.Equal("Sub System.Console.WriteLine(value As System.Boolean)", sortedMethodGroup(8).ToTestDisplayString())
            Assert.Equal("Sub System.Console.WriteLine(value As System.Char)", sortedMethodGroup(9).ToTestDisplayString())
            Assert.Equal("Sub System.Console.WriteLine(value As System.Decimal)", sortedMethodGroup(10).ToTestDisplayString())
            Assert.Equal("Sub System.Console.WriteLine(value As System.Double)", sortedMethodGroup(11).ToTestDisplayString())
            Assert.Equal("Sub System.Console.WriteLine(value As System.Int32)", sortedMethodGroup(12).ToTestDisplayString())
            Assert.Equal("Sub System.Console.WriteLine(value As System.Int64)", sortedMethodGroup(13).ToTestDisplayString())
            Assert.Equal("Sub System.Console.WriteLine(value As System.Object)", sortedMethodGroup(14).ToTestDisplayString())
            Assert.Equal("Sub System.Console.WriteLine(value As System.Single)", sortedMethodGroup(15).ToTestDisplayString())
            Assert.Equal("Sub System.Console.WriteLine(value As System.String)", sortedMethodGroup(16).ToTestDisplayString())
            Assert.Equal("Sub System.Console.WriteLine(value As System.UInt32)", sortedMethodGroup(17).ToTestDisplayString())
            Assert.Equal("Sub System.Console.WriteLine(value As System.UInt64)", sortedMethodGroup(18).ToTestDisplayString())

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub ReturnedNothingLiteral()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System

Module Program
    Function Main(args As String()) As String
        Return Nothing'BIND:"Nothing"
    End Function
End Module
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of LiteralExpressionSyntax)(compilation, "a.vb")

            Assert.Null(semanticInfo.Type)
            Assert.Equal("System.String", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.WideningNothingLiteral, semanticInfo.ImplicitConversion.Kind)

            Assert.Null(semanticInfo.Symbol)
            Assert.Equal(CandidateReason.None, semanticInfo.CandidateReason)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.True(semanticInfo.ConstantValue.HasValue)
            Assert.Null(semanticInfo.ConstantValue.Value)
        End Sub

        <ConditionalFact(GetType(WindowsOnly), Reason:="https://github.com/dotnet/roslyn/issues/29531")>
        Public Sub FailedConstructorCall()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System

Class C
End Class

Class V
    Sub goo
        Dim c As C
        c = New C(13)'BIND:"C"
    End Sub
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Null(semanticInfo.Type)
            Assert.Null(semanticInfo.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("C", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(CandidateReason.None, semanticInfo.CandidateReason)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <ConditionalFact(GetType(WindowsOnly), Reason:="https://github.com/dotnet/roslyn/issues/29531")>
        Public Sub FailedConstructorCall2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System

Class C
End Class

Class V
    Sub goo
        Dim c As C
        c = New C(13)'BIND:"New C(13)"
    End Sub
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of ObjectCreationExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("C", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("C", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Null(semanticInfo.Symbol)
            Assert.Equal(CandidateReason.OverloadResolutionFailure, semanticInfo.CandidateReason)
            Assert.Equal(1, semanticInfo.CandidateSymbols.Length)
            Dim sortedCandidates = semanticInfo.CandidateSymbols.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Sub C..ctor()", sortedCandidates(0).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(0).Kind)

            Assert.Equal(1, semanticInfo.MemberGroup.Length)
            Dim sortedMethodGroup = semanticInfo.MemberGroup.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Sub C..ctor()", sortedMethodGroup(0).ToTestDisplayString())

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact>
        Public Sub ExplicitCallToDefaultProperty1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Class X
    Public ReadOnly Property Goo As Y
        Get
            Return Nothing
        End Get
    End Property
End Class

Class Y
    Public Default ReadOnly Property Item(ByVal a As Integer) As String
        Get
            Return "hi"
        End Get
    End Property
End Class

Module M1
    Sub Main()
        Dim a As String
        Dim b As X
        b = New X()
        a = b.Goo.Item(4)'BIND:"Item"
    End Sub
End Module
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Null(semanticInfo.Type)
            Assert.Null(semanticInfo.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("ReadOnly Property Y.Item(a As System.Int32) As System.String", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Property, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(1, semanticInfo.MemberGroup.Length)
            Dim sortedMethodGroup = semanticInfo.MemberGroup.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("ReadOnly Property Y.Item(a As System.Int32) As System.String", sortedMethodGroup(0).ToTestDisplayString())

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact()>
        Public Sub ExplicitCallToDefaultProperty2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Class X
    Public ReadOnly Property Goo As Y
        Get
            Return Nothing
        End Get
    End Property
End Class

Class Y
    Public Default ReadOnly Property Item(ByVal a As Integer) As String
        Get
            Return "hi"
        End Get
    End Property
End Class

Module M1
    Sub Main()
        Dim a As String
        Dim b As X
        b = New X()
        a = b.Goo.Item(4)'BIND:"b.Goo.Item(4)"
    End Sub
End Module
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of InvocationExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("System.String", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("System.String", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("ReadOnly Property Y.Item(a As System.Int32) As System.String", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Property, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <WorkItem(541240, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/541240")>
        <Fact()>
        Public Sub ConstFieldInitializer()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Class C
    Public Const ClassId As String = Nothing 'BIND:"Nothing"
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of ExpressionSyntax)(compilation, "a.vb")

            Assert.Null(semanticInfo.Type)
            Assert.True(semanticInfo.ConstantValue.HasValue)
            Assert.Null(semanticInfo.ConstantValue.Value)
        End Sub

        <Fact()>
        Public Sub ConstFieldInitializer2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Class C
    Public Const ClassId As Integer = 23 'BIND:"23"
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of ExpressionSyntax)(compilation, "a.vb")

            Assert.NotNull(semanticInfo.Type)
            Assert.Equal(System_Int32, semanticInfo.Type.SpecialType)
            Assert.Equal(semanticInfo.ConstantValue.Value, 23)
            Assert.True(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact()>
        Public Sub ConstFieldInitializer3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Class C
    Public Const ClassDate As DateTime = #11/04/2008# 'BIND:"#11/04/2008#"
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of ExpressionSyntax)(compilation, "a.vb")

            Assert.NotNull(semanticInfo.Type)
            Assert.Equal(System_DateTime, semanticInfo.Type.SpecialType)
            Assert.True(semanticInfo.ConstantValue.HasValue)
            Assert.Equal(#11/4/2008#, semanticInfo.ConstantValue.Value)
        End Sub

        <Fact()>
        Public Sub ConstFieldInitializer4()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Class C
    Public Const ClassId As Integer = 2 + 2 'BIND:"2 + 2"
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of ExpressionSyntax)(compilation, "a.vb")

            Assert.NotNull(semanticInfo.Type)
            Assert.Equal(System_Int32, semanticInfo.Type.SpecialType)
            Assert.Equal(semanticInfo.ConstantValue.Value, 4)
            Assert.True(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact()>
        Public Sub ConstFieldInitializer5()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Class C
    Public Const A As Integer = 4
    Public Const B As Integer = 7 + 2 * A 'BIND:"2 * A"
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of ExpressionSyntax)(compilation, "a.vb")

            Assert.NotNull(semanticInfo.Type)
            Assert.Equal(System_Int32, semanticInfo.Type.SpecialType)
            Assert.Equal(semanticInfo.ConstantValue.Value, 8)
            Assert.True(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact()>
        Public Sub ConstFieldInitializersMultipleSymbols()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Class C
    Const X, Y As Integer = 6
    Const Z As Integer = Y
End Class
    </file>
</compilation>)
            Dim tree = compilation.SyntaxTrees(0)
            Dim model = compilation.GetSemanticModel(tree)

            Dim node = FindNodeFromText(tree, "X")
            Dim symbol = DirectCast(model.GetDeclaredSymbol(node), FieldSymbol)
            Assert.Equal(symbol.Name, "X")
            Assert.Equal(System_Int32, symbol.Type.SpecialType)
            Assert.Equal(symbol.ConstantValue, 6)

        End Sub

        <Fact()>
        Public Sub FieldInitializer1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Class C
    Public ClassId As Integer = 23 'BIND:"23"
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of ExpressionSyntax)(compilation, "a.vb")

            Assert.NotNull(semanticInfo.Type)
            Assert.Equal(System_Int32, semanticInfo.Type.SpecialType)
            Assert.Equal(semanticInfo.ConstantValue.Value, 23)
            Assert.True(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact()>
        Public Sub FieldInitializer2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Class C
    Public ClassId As Integer = C.goo() 'BIND:"goo"

    shared Function goo() as Integer
        return 23    
    End Function
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of ExpressionSyntax)(compilation, "a.vb")

            Assert.Null(semanticInfo.Type)
            Assert.Equal(1, semanticInfo.MemberGroup.Length)
            Assert.Equal("Function C.goo() As System.Int32", semanticInfo.MemberGroup(0).ToDisplayString(SymbolDisplayFormat.TestFormat))
        End Sub

        <WorkItem(541243, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/541243")>
        <Fact()>
        Public Sub CollectionInitializerNoMscorlibRef()
            Dim compilation = CompilationUtils.CreateEmptyCompilationWithReferences(
<compilation>
    <file name="a.vb">
Module Module1
    Sub Main()
        Dim numbers() As Integer = New Integer() {0, 1, 2, 3, 4} 'BIND:"{0, 1, 2, 3, 4}"
    End Sub
End Module        
    </file>
</compilation>, {})

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of ExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("System.Int32[missing]()", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Array, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Int32[missing]()", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Array, semanticInfo.ConvertedType.TypeKind)
        End Sub

        <Fact()>
        Public Sub CollectionInitializerWithoutConversion()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Option Strict On
Class C
    Shared Sub M(s as string)
        Dim s = New String() {s} 'BIND:"{s}"
    End Sub
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of ExpressionSyntax)(compilation, "a.vb")
            Assert.Equal("System.String()", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Array, semanticInfo.Type.TypeKind)
            Assert.Equal("System.String()", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Array, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)
        End Sub

        <WorkItem(541422, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/541422")>
        <Fact()>
        Public Sub CollectionInitializerWithConversion()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Option Strict On
Class C
    Shared Sub M(o As Object)
        Dim s = New String() {o} 'BIND:"{o}"
    End Sub
End Class
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of ExpressionSyntax)(compilation, "a.vb")
            Assert.Equal("System.String()", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Array, semanticInfo.Type.TypeKind)
            Assert.Equal("System.String()", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Array, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)
        End Sub

        <Fact()>
        Public Sub ArrayInitializerMemberWithConversion()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Imports System
Imports System.Collections.Generic
Imports System.Linq

Module Program
    Sub Main(args As String())
        Dim s As String
        Dim x As Object() = New Object() {s}'BIND:"s"
    End Sub
End Module
    ]]></file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("System.String", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Object", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.WideningReference, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("s As System.String", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Local, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact()>
        Public Sub TwoDArrayInitializerMember()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Imports System

Module Program
    Sub Main()
        Dim s1 As String = "hi"
        Dim s2 As String = "hello"
        Dim o1 As Object = Nothing
        Dim o2 As Object = Nothing
        Dim arr As Object(,) = New Object(,) {{o1, o2}, {s1, s2}}'BIND:"s2"
    End Sub
End Module
    ]]></file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("System.String", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Object", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.WideningReference, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("s2 As System.String", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Local, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact()>
        Public Sub PartialArrayInitializer()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Imports System

Module Program
    Sub Main()
        Dim s1 As String = "hi"
        Dim s2 As String = "hello"
        Dim o1 As Object = Nothing
        Dim o2 As Object = Nothing
        Dim arr As Object(,) = New Object(,) {{o1, o2}, {s1, s2}}'BIND:"{s1, s2}"
    End Sub
End Module
    ]]></file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of CollectionInitializerSyntax)(compilation, "a.vb")

            Assert.Null(semanticInfo.Type)
            Assert.Null(semanticInfo.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Null(semanticInfo.Symbol)
            Assert.Equal(CandidateReason.None, semanticInfo.CandidateReason)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <WorkItem(541270, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/541270")>
        <Fact()>
        Public Sub GetSemanticInfoOfNothing()
            Dim compilation = CompilationUtils.CreateEmptyCompilationWithReferences(
<compilation>
    <file name="a.vb">
Module Module1
    Sub Main()
    End Sub
End Module        
    </file>
</compilation>, {})

            Dim semanticModel = compilation.GetSemanticModel(compilation.SyntaxTrees(0))

            Assert.Throws(Of ArgumentNullException)(Function()
                                                        Return semanticModel.GetAliasInfo(Nothing)
                                                    End Function
            )

            Assert.Throws(Of ArgumentNullException)(Function()
                                                        Return semanticModel.GetSymbolInfo(DirectCast(Nothing, AttributeSyntax))
                                                    End Function
            )

            Assert.Throws(Of ArgumentNullException)(Function()
                                                        Return semanticModel.GetSymbolInfo(DirectCast(Nothing, ExpressionRangeVariableSyntax))
                                                    End Function
            )

            Assert.Throws(Of ArgumentNullException)(Function()
                                                        Return semanticModel.GetSymbolInfo(DirectCast(Nothing, ExpressionSyntax))
                                                    End Function
            )

            Assert.Throws(Of ArgumentNullException)(Function()
                                                        Return semanticModel.GetSymbolInfo(DirectCast(Nothing, FunctionAggregationSyntax))
                                                    End Function
            )

            Assert.Throws(Of ArgumentNullException)(Function()
                                                        Return semanticModel.GetSymbolInfo(DirectCast(Nothing, OrderingSyntax))
                                                    End Function
            )

            Assert.Throws(Of ArgumentNullException)(Function()
                                                        Return semanticModel.GetSymbolInfo(DirectCast(Nothing, QueryClauseSyntax))
                                                    End Function
            )

            Assert.Throws(Of ArgumentNullException)(Function()
                                                        Return semanticModel.GetTypeInfo(DirectCast(Nothing, AttributeSyntax))
                                                    End Function
            )

            Assert.Throws(Of ArgumentNullException)(Function()
                                                        Return semanticModel.GetTypeInfo(DirectCast(Nothing, ExpressionSyntax))
                                                    End Function
            )

            Assert.Throws(Of ArgumentNullException)(Function()
                                                        Return semanticModel.GetConstantValue(DirectCast(Nothing, ExpressionSyntax))
                                                    End Function
            )

            Assert.Throws(Of ArgumentNullException)(Function()
                                                        Return semanticModel.GetMemberGroup(DirectCast(Nothing, ExpressionSyntax))
                                                    End Function
            )

        End Sub

        <WorkItem(541390, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/541390")>
        <Fact()>
        Public Sub ErrorLambdaParamInsideFieldInitializer()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation>
    <file name="a.vb">
Module Module1
    Dim f1 As Func(Of Integer, Integer) = Function(lambdaParam As Integer)
                                              Return lambdaParam + 1 'BIND:"lambdaParam"
                                          End Function
End Module      
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of ExpressionSyntax)(compilation, "a.vb")

            Assert.NotNull(semanticInfo)
            Assert.NotNull(semanticInfo.Type)
            Assert.Equal(System_Int32, semanticInfo.Type.SpecialType)
        End Sub

        <Fact()>
        Public Sub ErrorLambdaParamInsideLocalInitializer()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation>
    <file name="a.vb">
Module Module1
    Sub Main()
        Dim f1 As Func(Of Integer, Integer) = Function(lambdaParam As Integer)
                                                  Return lambdaParam + 1 'BIND:"lambdaParam"
                                              End Function
    End Sub
End Module      
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of ExpressionSyntax)(compilation, "a.vb")

            Assert.NotNull(semanticInfo)
            Assert.NotNull(semanticInfo.Type)
            Assert.Equal(System_Int32, semanticInfo.Type.SpecialType)
        End Sub

        <WorkItem(541390, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/541390")>
        <Fact()>
        Public Sub LambdaParamInsideFieldInitializer()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation>
    <file name="a.vb">
Imports System        
Module Module1
    Dim f1 As Func(Of Integer, Integer) = Function(lambdaParam)
                                              Return lambdaParam + 1 'BIND:"lambdaParam"
                                          End Function
End Module      
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of ExpressionSyntax)(compilation, "a.vb")

            Assert.NotNull(semanticInfo)
            Assert.NotNull(semanticInfo.Type)
            Assert.Equal(System_Int32, semanticInfo.Type.SpecialType)
        End Sub

        <Fact()>
        Public Sub LambdaParamInsideLocalInitializer()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation>
    <file name="a.vb">
Imports System
Module Module1
    Sub Main()
        Dim f1 As Func(Of Integer, Integer) = Function(lambdaParam)
                                                  Return lambdaParam + 1 'BIND:"lambdaParam"
                                              End Function
    End Sub
End Module      
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of ExpressionSyntax)(compilation, "a.vb")

            Assert.NotNull(semanticInfo)
            Assert.NotNull(semanticInfo.Type)
            Assert.Equal(System_Int32, semanticInfo.Type.SpecialType)
        End Sub

        <WorkItem(541418, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/541418")>
        <Fact()>
        Public Sub BindAttributeInstanceWithoutAttributeSuffix()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation>
    <file name="a.vb"><![CDATA[
<Assembly: My> 'BIND:"My"
Class MyAttribute : Inherits System.Attribute
End Class]]>
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of ExpressionSyntax)(compilation, "a.vb")

            Assert.NotNull(semanticInfo)
            Assert.NotNull(semanticInfo.Symbol)
            Assert.Equal("Sub MyAttribute..ctor()", semanticInfo.Symbol.ToDisplayString(SymbolDisplayFormat.TestFormat))
            Assert.Equal("MyAttribute", semanticInfo.Type.ToDisplayString(SymbolDisplayFormat.TestFormat))
        End Sub

        <WorkItem(541401, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/541401")>
        <Fact()>
        Public Sub BindingAttributeParameter()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Class MineAttribute

    Inherits Attribute
    Public Sub New(p As Short)
    End Sub

End Class

<Mine(123)> 'BIND:"123"
Class C
End Class
]]>
    </file>
</compilation>)

            ' NOTE: VB doesn't allow same line comments after attribute
            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of LiteralExpressionSyntax)(compilation, "a.vb")

            Assert.NotNull(semanticInfo.Type)
            Assert.Equal(System_Int32, semanticInfo.Type.SpecialType)
            Assert.Equal(System_Int16, semanticInfo.ConvertedType.SpecialType)
            Assert.Equal(semanticInfo.ConstantValue.Value, 123)
            Assert.True(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact()>
        Public Sub BindAttributeNamespace()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
        <![CDATA[
Imports System
Imports System.Collections.Generic
Imports System.Linq

Namespace n1

    Module Program

        <n1.Program.Test1(1)>'BIND:"n1"
        Class A
        End Class

        <AttributeUsageAttribute(AttributeTargets.All, AllowMultiple:=True)>
        Class Test1
            Inherits Attribute

              Public Sub New(i As Integer)
            End Sub

        End Class

    End Module

End Namespace
    ]]></file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Null(semanticInfo.Type)
            Assert.Null(semanticInfo.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("n1", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Namespace, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <WorkItem(541418, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/541418")>
        <Fact()>
        Public Sub BindingAttributeClassName()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Imports System
<System.AttributeUsage(AttributeTargets.All, AllowMultiple:=true)> _ 'BIND:"AttributeUsage"
Class ZAttribute
    Inherits Attribute

End Class
<ZAttribute()>
Class scen1
    Shared Sub Main()
        Dim x = 1
        Console.WriteLine(x)
    End Sub
End Class
]]>
    </file>
</compilation>)

            ' NOTE: VB doesn't allow same line comments after attribute
            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")
            Assert.NotNull(semanticInfo.Type)
            Assert.Equal("System.AttributeUsageAttribute", semanticInfo.Type.ToString())
            Assert.False(DirectCast(semanticInfo.Type, TypeSymbol).IsErrorType)
        End Sub

        <Fact()>
        Public Sub TestAttributeFieldName()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Imports System
Imports System.Collections.Generic
Imports System.Linq

Namespace n1

    Module Program

        <Test1(fi:=10)>'BIND:"fi"
        Class A
        End Class

    <AttributeUsageAttribute(AttributeTargets.All, AllowMultiple:=True)>
    Class Test1
        Inherits Attribute

        Public Sub New(i As Integer)
        End Sub

        Public fi As Integer

    End Class

    End Module

End Namespace]]>
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("System.Int32", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Int32", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("n1.Program.Test1.fi As System.Int32", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Field, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact()>
        Public Sub TestAttributePropertyName()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Imports System
Imports System.Collections.Generic
Imports System.Linq

Namespace n1

    Module Program

        Class A
            <Test1(1, Pi:=2)>'BIND:"Pi"
            Sub s
            End Sub

        End Class

    <AttributeUsageAttribute(AttributeTargets.All, AllowMultiple:=True)>
    Class Test1
        Inherits Attribute

        Public Sub New()
        End Sub

        Public Sub New(i As Integer)
            End Sub

            Public fi As Integer

            Public Property Pi As Integer

        End Class

    End Module

End Namespace
    ]]></file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("System.Int32", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Int32", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Property n1.Program.Test1.Pi As System.Int32", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Property, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact()>
        Public Sub TestAttributePositionalArgOnParameter()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Imports System
Imports System.Collections.Generic
Imports System.Linq

Namespace n1

    Module Program

        Class A
            Function f( <Test1("parameter")> x As Integer) As Integer'BIND:""parameter""
                Return 0
            End Function
        End Class

    <AttributeUsageAttribute(AttributeTargets.All, AllowMultiple:=True)>
    Class Test1
        Inherits Attribute

            Public Sub New(i As String)
            End Sub

        End Class

    End Module

End Namespace
    ]]></file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of LiteralExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("System.String", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("System.String", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Null(semanticInfo.Symbol)
            Assert.Equal(CandidateReason.None, semanticInfo.CandidateReason)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.True(semanticInfo.ConstantValue.HasValue)
            Assert.Equal("parameter", semanticInfo.ConstantValue.Value)
        End Sub

        <Fact()>
        Public Sub TestAttributeClassNameOnReturnValue()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Imports System
Imports System.Collections.Generic
Imports System.Linq

Namespace n1

    Module Program

        Class A

            Function f(x As Integer) As <Test1(4)> Integer'BIND:"Test1"
                Return 0
            End Function
        End Class

    <AttributeUsageAttribute(AttributeTargets.All, AllowMultiple:=True)>
    Class Test1
        Inherits Attribute

        Public Sub New(i As Integer)
        End Sub

        End Class

    End Module

End Namespace
    ]]></file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("n1.Program.Test1", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("n1.Program.Test1", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Sub n1.Program.Test1..ctor(i As System.Int32)", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(1, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact()>
        Public Sub TestAttributeCannotBindToUnqualifiedClassMember()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Imports System
Imports System.Collections.Generic
Imports System.Linq

Namespace n1

    Module Program

        <Test1(C1)>'BIND:"C1"
        Class A
            Public Const C1 As Integer = 99
        End Class

    <AttributeUsageAttribute(AttributeTargets.All, AllowMultiple:=True)>
    Class Test1
        Inherits Attribute

            Public Sub New(i As Integer)
            End Sub

        End Class

    End Module

End Namespace
    ]]></file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("?", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Error, semanticInfo.Type.TypeKind)
            Assert.Equal("?", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Error, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Null(semanticInfo.Symbol)
            Assert.Equal(CandidateReason.None, semanticInfo.CandidateReason)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <ConditionalFact(GetType(WindowsOnly), Reason:="https://github.com/dotnet/roslyn/issues/29531")>
        Public Sub AttributeSemanticInfo_OverloadResolutionFailure_01()
            Dim compilation = CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Imports System
<Module: ObsoleteAttribute(GetType())>'BIND:"ObsoleteAttribute"
    ]]></file>
</compilation>)

            Dim semanticSummary = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("System.ObsoleteAttribute", semanticSummary.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticSummary.Type.TypeKind)
            Assert.Equal("System.ObsoleteAttribute", semanticSummary.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticSummary.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticSummary.ImplicitConversion.Kind)

            Assert.Null(semanticSummary.Symbol)
            Assert.Equal(CandidateReason.OverloadResolutionFailure, semanticSummary.CandidateReason)
            Assert.Equal(3, semanticSummary.CandidateSymbols.Length)
            Dim sortedCandidates = semanticSummary.CandidateSymbols.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString(), StringComparer.Ordinal).ToArray()
            Assert.Equal("Sub System.ObsoleteAttribute..ctor()", sortedCandidates(0).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(0).Kind)
            Assert.Equal("Sub System.ObsoleteAttribute..ctor(message As System.String)", sortedCandidates(1).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(1).Kind)
            Assert.Equal("Sub System.ObsoleteAttribute..ctor(message As System.String, [error] As System.Boolean)", sortedCandidates(2).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(2).Kind)

            Assert.Null(semanticSummary.Alias)

            Assert.Equal(3, semanticSummary.MemberGroup.Length)
            Dim sortedMethodGroup = semanticSummary.MemberGroup.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString(), StringComparer.Ordinal).ToArray()
            Assert.Equal("Sub System.ObsoleteAttribute..ctor()", sortedMethodGroup(0).ToTestDisplayString())
            Assert.Equal("Sub System.ObsoleteAttribute..ctor(message As System.String)", sortedMethodGroup(1).ToTestDisplayString())
            Assert.Equal("Sub System.ObsoleteAttribute..ctor(message As System.String, [error] As System.Boolean)", sortedMethodGroup(2).ToTestDisplayString())

            Assert.False(semanticSummary.ConstantValue.HasValue)
        End Sub

        <ConditionalFact(GetType(WindowsOnly), Reason:="https://github.com/dotnet/roslyn/issues/29531")>
        Public Sub AttributeSemanticInfo_OverloadResolutionFailure_02()
            Dim compilation = CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Imports System
<Module: Obsolete(GetType())>'BIND:"Module: Obsolete(GetType())"
    ]]></file>
</compilation>)

            Dim semanticSummary = CompilationUtils.GetSemanticInfoSummary(Of AttributeSyntax)(compilation, "a.vb")

            Assert.Equal("System.ObsoleteAttribute", semanticSummary.Type.ToTestDisplayString())
            Assert.Equal("System.ObsoleteAttribute", semanticSummary.ConvertedType.ToTestDisplayString())
            Assert.Equal(ConversionKind.Identity, semanticSummary.ImplicitConversion.Kind)

            Assert.Null(semanticSummary.Symbol)
            Assert.Equal(CandidateReason.OverloadResolutionFailure, semanticSummary.CandidateReason)
            Assert.Equal(3, semanticSummary.CandidateSymbols.Length)
            Dim sortedCandidates = semanticSummary.CandidateSymbols.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString(), StringComparer.Ordinal).ToArray()
            Assert.Equal("Sub System.ObsoleteAttribute..ctor()", sortedCandidates(0).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(0).Kind)
            Assert.Equal("Sub System.ObsoleteAttribute..ctor(message As System.String)", sortedCandidates(1).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(1).Kind)
            Assert.Equal("Sub System.ObsoleteAttribute..ctor(message As System.String, [error] As System.Boolean)", sortedCandidates(2).ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, sortedCandidates(2).Kind)

            Assert.Null(semanticSummary.Alias)

            Assert.Equal(3, semanticSummary.MemberGroup.Length)
            Dim sortedMethodGroup = semanticSummary.MemberGroup.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString(), StringComparer.Ordinal).ToArray()
            Assert.Equal("Sub System.ObsoleteAttribute..ctor()", sortedMethodGroup(0).ToTestDisplayString())
            Assert.Equal("Sub System.ObsoleteAttribute..ctor(message As System.String)", sortedMethodGroup(1).ToTestDisplayString())
            Assert.Equal("Sub System.ObsoleteAttribute..ctor(message As System.String, [error] As System.Boolean)", sortedMethodGroup(2).ToTestDisplayString())

            Assert.False(semanticSummary.ConstantValue.HasValue)
        End Sub

        <WorkItem(541481, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/541481")>
        <Fact()>
        Public Sub BindingPredefinedCastExpression()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation>
    <file name="a.vb"><![CDATA[
Imports Microsoft.VisualBasic

Module Test

    Sub Main()
        Dim exp As Integer = 123
        Dim act As String = CStr(exp) 'BIND:"CStr(exp)"
    End Sub

End Module
]]>
    </file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of PredefinedCastExpressionSyntax)(compilation, "a.vb")
            Assert.NotNull(semanticInfo.Type)
            Assert.Equal("String", semanticInfo.Type.ToString())
        End Sub

        <WorkItem(541498, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/541498")>
        <Fact()>
        Public Sub DictionaryAccessExpressionErrorType()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation>
    <file name="a.vb"><![CDATA[
Module Module1
    Sub Main()
        Dim c As Collection
        Dim b = c!A 'BIND:"c!A"
    End Sub
End Module
]]>
    </file>
</compilation>)
            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of MemberAccessExpressionSyntax)(compilation, "a.vb")
            Assert.NotNull(semanticInfo)
            Assert.Null(semanticInfo.Symbol)
            Assert.Equal(semanticInfo.Type.TypeKind, TypeKind.Error)
        End Sub

        <Fact()>
        Public Sub DictionaryAccessExpressionErrorExpr()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation>
    <file name="a.vb"><![CDATA[
Class A
    Private Shared F As System.Collections.Generic.Dictionary(Of String, Integer)
End Class
Class B
    Shared Sub M()
        Dim o = A.F!x 'BIND:"A.F!x"
    End Sub
End Class
]]>
    </file>
</compilation>)
            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of MemberAccessExpressionSyntax)(compilation, "a.vb")
            Assert.NotNull(semanticInfo)
            CompilationUtils.CheckSymbol(semanticInfo.Symbol, "Property Dictionary(Of String, Integer).Item(key As String) As Integer")
            Assert.Equal(semanticInfo.Type.SpecialType, System_Int32)
        End Sub

        <Fact()>
        Public Sub DictionaryAccessExpressionNoType()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation>
    <file name="a.vb"><![CDATA[
Class C
    Shared Sub M()
        Dim o = (AddressOf M)!x 'BIND:"(AddressOf M)!x"
    End Sub
End Class
]]>
    </file>
</compilation>)
            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of MemberAccessExpressionSyntax)(compilation, "a.vb")
            Assert.NotNull(semanticInfo)
            Assert.Null(semanticInfo.Symbol)
            Assert.Equal(semanticInfo.Type.TypeKind, TypeKind.Error)
        End Sub

        <Fact()>
        Public Sub DictionaryAccessExpression()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation>
    <file name="a.vb"><![CDATA[
Class A
    Friend Shared F As System.Collections.Generic.Dictionary(Of String, Integer)
End Class
Class B
    Shared Sub M()
        Dim o = A.F!x 'BIND:"A.F!x"
    End Sub
End Class
]]>
    </file>
</compilation>)
            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of MemberAccessExpressionSyntax)(compilation, "a.vb")
            Assert.NotNull(semanticInfo)
            CompilationUtils.CheckSymbol(semanticInfo.Symbol, "Property Dictionary(Of String, Integer).Item(key As String) As Integer")
            Assert.Equal(semanticInfo.Type.SpecialType, System_Int32)
        End Sub

        <WorkItem(541384, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/541384")>
        <Fact()>
        Public Sub DictionaryAccessKey()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation>
    <file name="a.vb"><![CDATA[
Class C
    Shared Function M(d As System.Collections.Generic.Dictionary(Of String, Integer)) As Integer
        Return d!key 'BIND:"key"
    End Function
End Class
]]>
    </file>
</compilation>)
            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")
            Assert.NotNull(semanticInfo)
            Assert.NotNull(semanticInfo.Type)
            Assert.Equal(semanticInfo.Type.SpecialType, System_String)
            Assert.Equal(semanticInfo.ConstantValue.Value, "key")
            Assert.Null(semanticInfo.Symbol)
        End Sub

        <WorkItem(541518, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/541518")>
        <Fact()>
        Public Sub AssignAddressOfPropertyToDelegate()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Module Module1
    Delegate Function del2() As Integer
    Property p() As Integer
        Get
            Return 10
        End Get
        Set(ByVal Value As Integer)

        End Set
    End Property

    Sub Main()
        Dim var2 = New del2(AddressOf p)'BIND:"p"
    End Sub
End Module
    ]]></file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Null(semanticInfo.Type)
            Assert.Null(semanticInfo.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Null(semanticInfo.Symbol)
            Assert.Equal(CandidateReason.OverloadResolutionFailure, semanticInfo.CandidateReason)
            Assert.Equal(1, semanticInfo.CandidateSymbols.Length)
            Dim sortedCandidates = semanticInfo.CandidateSymbols.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Property Module1.p As System.Int32", sortedCandidates(0).ToTestDisplayString())
            Assert.Equal(SymbolKind.Property, sortedCandidates(0).Kind)

            Assert.Equal(1, semanticInfo.MemberGroup.Length)
            Dim sortedMethodGroup = semanticInfo.MemberGroup.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Property Module1.p As System.Int32", sortedMethodGroup(0).ToTestDisplayString())

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact()>
        Public Sub FieldAccess()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Imports System

Class B
    Public f1 As Integer
End Class

Class M
    Public Sub Main()
        Dim bInstance As B
        bInstance = New B()
        Console.WriteLine(bInstance.f1)'BIND:"f1"
    End Sub
End Class
    ]]></file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("System.Int32", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Int32", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("B.f1 As System.Int32", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Field, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact()>
        Public Sub LocalVariable()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Imports System

Class B
    Public f1 As Integer
End Class

Class M
    Public Sub Main()
        Dim bInstance As B
        bInstance = New B()
        Console.WriteLine(bInstance.f1)'BIND:"bInstance"
    End Sub
End Class
    ]]></file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("B", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("B", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("bInstance As B", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Local, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact()>
        Public Sub NotFunctionReturnLocal()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Imports System

Class M
    Public Function Goo() As Integer
        Goo()'BIND:"Goo"
    End Function
End Class
    ]]></file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Null(semanticInfo.Type)
            Assert.Null(semanticInfo.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Function M.Goo() As System.Int32", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(1, semanticInfo.MemberGroup.Length)
            Dim sortedMethodGroup = semanticInfo.MemberGroup.AsEnumerable().OrderBy(Function(s) s.ToTestDisplayString()).ToArray()
            Assert.Equal("Function M.Goo() As System.Int32", sortedMethodGroup(0).ToTestDisplayString())

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact()>
        Public Sub FunctionReturnLocal()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Imports System

Class M
    Public Function Goo() As Integer
        Goo = 4'BIND:"Goo"
    End Function
End Class

    ]]></file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("System.Int32", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Int32", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Goo As System.Int32", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Local, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact()>
        Public Sub MeSemanticInfo()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Imports System

Class M
    Public x As Integer
    Public Function Goo() As Integer
        Me.x = 5'BIND:"Me"
    End Function
End Class
    ]]></file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of MeExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("M", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("M", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Me As M", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Parameter, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.True(DirectCast(semanticInfo.Symbol, ParameterSymbol).IsMe)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact()>
        Public Sub LocalVariableInConstructor()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Imports System

Class B
    Public f1 As Integer
End Class

Class M
    Public Sub New()
        Dim bInstance As B
        bInstance = New B()
        Console.WriteLine(bInstance.f1)'BIND:"bInstance"
    End Sub
End Class

    ]]></file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("B", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("B", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("bInstance As B", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Local, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact()>
        Public Sub LocalVariableInSharedConstructor()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Imports System

Class B
    Public f1 As Integer
End Class

Class M
    Shared Sub New()
        Dim bInstance As B
        bInstance = New B()
        Console.WriteLine(bInstance.f1)'BIND:"bInstance"
    End Sub
End Class
    ]]></file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("B", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("B", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("bInstance As B", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Local, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact()>
        Public Sub LocalVariableInModuleConstructor()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Imports System

Class B
    Public f1 As Integer
End Class

Module M
    Sub New()
        Dim bInstance As B
        bInstance = New B()
        Console.WriteLine(bInstance.f1)'BIND:"bInstance"
    End Sub
End Module

    ]]></file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Equal("B", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("B", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("bInstance As B", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Local, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact()>
        Public Sub ConstructorConstructorCall_Structure_Me_New_WithParameters()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Imports System
Structure Program
    Public Sub New(i As Integer)
    End Sub

    Public Sub New(s As String)
        Me.New(1)'BIND:"Me.New(1)"
    End Sub
End Structure
    ]]></file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of InvocationExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("System.Void", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Void", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Sub Program..ctor(i As System.Int32)", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)

            Dim method = DirectCast(semanticInfo.Symbol, MethodSymbol)
            Assert.Equal(MethodKind.Constructor, method.MethodKind)
            Assert.False(method.IsShared)
            Assert.False(method.IsImplicitlyDeclared)
        End Sub

        <Fact()>
        Public Sub ConstructorConstructorCall_Class_Me_New_WithParameters()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Imports System
Class Program
    Public Sub New(i As Integer)
    End Sub

    Public Sub New(s As String)
        Me.New(1)'BIND:"Me.New(1)"
    End Sub
End Class
    ]]></file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of InvocationExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("System.Void", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Void", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Sub Program..ctor(i As System.Int32)", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)

            Dim method = DirectCast(semanticInfo.Symbol, MethodSymbol)
            Assert.Equal(MethodKind.Constructor, method.MethodKind)
            Assert.False(method.IsShared)
            Assert.False(method.IsImplicitlyDeclared)
        End Sub

        <Fact()>
        Public Sub ConstructorConstructorCall_Structure_Me_New_NoParameters()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Imports System
Structure Program
    Public Sub New(s As String)
        Me.New()'BIND:"Me.New()"
    End Sub
End Structure
    ]]></file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of InvocationExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("System.Void", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Void", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Sub Program..ctor()", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)

            Dim method = DirectCast(semanticInfo.Symbol, MethodSymbol)
            Assert.Equal(MethodKind.Constructor, method.MethodKind)
            Assert.False(method.IsShared)
            Assert.True(method.IsImplicitlyDeclared)
        End Sub

        <Fact()>
        Public Sub ConstructorConstructorCall_Class_Me_New_NoParameters()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Imports System
Class Program
    Public Sub New()
    End Sub

    Public Sub New(s As String)
        Me.New()'BIND:"Me.New()"
    End Sub
End Class
    ]]></file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of InvocationExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("System.Void", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.Type.TypeKind)
            Assert.Equal("System.Void", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Structure, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Sub Program..ctor()", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)

            Dim method = DirectCast(semanticInfo.Symbol, MethodSymbol)
            Assert.Equal(MethodKind.Constructor, method.MethodKind)
            Assert.False(method.IsShared)
            Assert.False(method.IsImplicitlyDeclared)
        End Sub

        <Fact()>
        Public Sub Invocation()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Class C
    Private Sub M()
        Dim x As String = F()'BIND:"F()"
    End Sub

    Private Function F() As String
        Return "Hello"
    End Function
End Class
    ]]></file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of InvocationExpressionSyntax)(compilation, "a.vb")

            Assert.Equal("System.String", semanticInfo.Type.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.Type.TypeKind)
            Assert.Equal("System.String", semanticInfo.ConvertedType.ToTestDisplayString())
            Assert.Equal(TypeKind.Class, semanticInfo.ConvertedType.TypeKind)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Function C.F() As System.String", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, semanticInfo.Symbol.Kind)
            Assert.Equal(0, semanticInfo.CandidateSymbols.Length)

            Assert.Equal(0, semanticInfo.MemberGroup.Length)

            Assert.False(semanticInfo.ConstantValue.HasValue)
        End Sub

        <Fact()>
        Public Sub MethodGroup()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb"><![CDATA[
Class C
    Private Sub M()
        Dim x As String = F()'BIND:"F"
    End Sub

    Private Function F() As String
        Return "Hello"
    End Function

    Private Function F(arg As String) As String
        Return "Goodbye"
    End Function
End Class
    ]]></file>
</compilation>)

            Dim semanticInfo = CompilationUtils.GetSemanticInfoSummary(Of IdentifierNameSyntax)(compilation, "a.vb")

            Assert.Null(semanticInfo.Type)
            Assert.Null(semanticInfo.ConvertedType)
            Assert.Equal(ConversionKind.Identity, semanticInfo.ImplicitConversion.Kind)

            Assert.Equal("Function C.F() As System.String", semanticInfo.Symbol.ToTestDisplayString())
            Assert.Equal(SymbolKind.Method, semanticInfo.Symbol.Kind)
            Assert.Equal(0, seman